-- --8<--8<--8<--8<--
--
-- Copyright (C) 2011 Smithsonian Astrophysical Observatory
--
-- This file is part of saotrace.raygen
--
-- saotrace.raygen is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--
-- -->8-->8-->8-->8--

local math = require( "math" )

local M = {}
setfenv( 1, M )

local      pi,         acos,      cos,      mod,
           atan2,      sin,       tan,      asin,
           atan,       sqrt
   =  math.pi,    math.acos, math.cos, math.mod,
      math.atan2, math.sin,  math.tan, math.asin,
      math.atan,  math.sqrt

sec2rad = pi / ( 180 * 3600 )
min2rad = pi / ( 180 * 60 )
deg2rad = pi / 180
rad2deg = 180 / pi

-----------------------------------------------------------------------------
-- Convert from OSAC elevation-azimuth to OSAC polar coordinates
--
-- @param el       elevation [radians]
-- @param az       azimuth   [radians]
--
-- @return         theta, phi [radians]
-----------------------------------------------------------------------------

function osac_elaz_2_osac_polar( el, az )

  local theta, phi

  el = mod( el, pi/2 )
  az = mod( az, pi/2 )

  -- since -pi/2 <= el <= pi/2 and -pi/2 <= az <= pi/2,
  -- cos(el) >= 0 and cos(az) >= 0, so theta >= 0
  theta = acos( cos( el ) * cos( az ) )

  -- atan2 returns -pi <= phi <= pi,
  -- tan(el) and sin(az) preserve the signs of el and az.
  phi = atan2( tan( el ), sin(az ) )

  -- convert phi s.t. 0 <= phi < 2*pi
  phi = mod( phi + 2 * pi, 2 * pi )

  return theta, phi

end

-----------------------------------------------------------------------------
-- Convert from OSAC polar coordinates to OSAC elevation-azimuth
--
-- @param theta    off-axis angle [radians]
-- @param phi      circum-polar angle   [radians]
--
-- @return         elevation, azimuth [radians]
-----------------------------------------------------------------------------

function osac_polar_2_osac_elaz( theta, phi )

  local el, az

  -- convert phi to 0 <= phi < 2*pi
  phi   = mod( mod( phi,   2 * pi ) + 2 * pi, 2 * pi)
  theta = mod( theta, pi / 2 ) 

  -- theta is always < $pi/2, so sin( theta ) > 0
  -- sign of sin(el) tracks sin(phi)
  el = asin( sin( theta ) * sin( phi ) )

  -- theta is always < $pi/2, so tan( theta ) > 0
  -- sign of az tracks cos(phi).
  az = atan( cos( phi ) * tan ( theta ) )

  return el, az
end

-- theta and phi are in radians
-- raygen polar is off by pi
function osac_polar_2_raygen_polar( theta, phi )

  phi = mod( phi + pi, 2 * pi )

  return theta, phi
end

-- convert from OSAC elevation and azimuth to X and Y at a particular Z
function osac_elaz_2_xy( el, az, z )

  local theta, phi = osac_elaz_2_osac_polar( el, az )
  phi = phi
  theta = theta

  local r = z * sin( theta )
  local y = r * sin( phi )
  local x = r * cos( phi )

  return x, y
end


-- Convert from B<raygen> polar coordinates to B<raygen> elevation and
-- azimuth.  All angles are in radians.  Input angles are reduced such
-- that 0 <= phi < 2pi and 0 <= theta <= pi/2

function raygen_polar_2_raygen_elaz( theta, phi )

  local el, az

  phi   = phi
  theta = theta

  -- convert phi to 0 <= phi < 2*pi, theta to 0 <= theta < pi/2
  phi   = mod( mod( phi, 2 * pi ) + 2 * pi, 2 * pi)
  theta = mod( theta, pi / 2 )

  -- theta is always < pi/2, so sin( theta ) > 0
  -- sign of sin(el) tracks sin(phi)
  el = asin( sin( theta ) * sin( phi ) )

  -- theta is always < pi/2, so tan( theta ) > 0
  -- sign of az tracks cos(phi).
  az = atan( cos( phi) * tan ( theta ) )

  return el, az
end



-- Convert from raygen elevation and azimuth to raygen polar coordinates.
-- All angles are in radians.  Input angles are reduced to be 0 <= angle
-- <= pi/2.

function raygen_elaz_2_raygen_polar( el, az )
  local theta, phi

  el = el
  az = az

  el = mod( el, 2 * pi )
  az = mod( az, 2 * pi )

  -- since -pi/2 <= el <= pi/2 and -pi/2 <= az <= pi/2,
  -- cos(el) >= 0 and cos(az) >= 0, so theta >= 0
  theta = acos( cos(el) * cos(az) )

  -- atan2 returns -pi <= phi <= pi,
  -- tan(el) and sin(az) preserve the signs of el and az.
  phi = atan2( tan(el), sin(az) )

  -- convert phi s.t. 0 <= phi < 2*pi
  phi   = mod( phi + 2 * pi, 2 * pi )

  return theta, phi

end

-- all angles are in radians
-- see http://en.wikipedia.org/wiki/Great-circle_distance

function delta_equatorial( ra_s, dec_s, ra_f, dec_f )

   local dra = ra_f - ra_s

   local x = (cos(dec_s) * sin (dec_f))  - (sin(dec_s) * cos(dec_f) * cos(dra))
   local y = cos(dec_f) * sin(dra)
   local z = (sin (dec_s) * sin (dec_f)) + (cos(dec_s) * cos(dec_f) * cos(dra))

   local theta = atan2(sqrt(x*x + y*y), z)

   y = sin(dra)
   x = (cos(dec_s) * tan(dec_f)) - (sin(dec_s) * cos(dra))

   local pa = atan2(y, x)

   return theta, pa
end


return M
