-- --8<--8<--8<--8<--
--
-- Copyright (C) 2011 Smithsonian Astrophysical Observatory
--
-- This file is part of saotrace.geometry
--
-- saotrace.geometry is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--
-- -->8-->8-->8-->8--

local require = require
local error   = error
local assert  = assert
local unpack  = unpack
local tonumber = tonumber

local coroutine = require( 'coroutine' )
local table = require('table')

require( 'string' )

local rdb = require('rdb')
local va = require( 'validate.args' )

module(...)

--- read a geometry database
-- @param file  the filename
-- @param mode  either 'iterator' or 'function'
--  uses named arguments
function read( ... )

   local ok, args = va.validate( {
				    file = { type = 'string' },
				    mode = { enum = { 'function',
						      'iterator',
						      'yield',
						   },
					     default = 'function',
					  },
				 },
				 ... )

   assert( ok, args )

   local yield
   local vars
   local next_entry

   if args.file:find( '.rdb$' ) then

      local fh = rdb( args.file )
      next_entry = function() return fh:read() end

      vars = fh.vars

   else

      error( args.file .. ": unknown file type\n" )

   end


   if args.mode == 'iterator' then

      local coroutine = require('coroutine')

      args.mode = 'yield'
      local co = coroutine.create( function() read( args ) end)
      local iter = function()
		      local res = { coroutine.resume( co ) }
		      if not res[1] then
			 error( unpack( res, 2  ) )
		      end

		      return unpack( res, 2 )
		   end

      return iter, vars

   elseif args.mode == 'yield' then

      yield = require('coroutine').yield

   end

   local entries = {}


   nentries = 0
   for entry in next_entry do

      nentries = nentries + 1

      -- fix up old tables
      if nil == entry.rthick then
	 entry.rthick = 0
      end

      if entry.mirror ~= nil and
	 ( entry.shell == nil or nil == entry.optic ) then

	 entry.optic, entry.shell = entry.mirror:match( "^([%a]+)(%d+)$" )
	 entry.shell = tonumber(entry.shell)
      end

      if args.mode == 'yield' then

	 yield( entry, nentries )

      else

	 table.insert( entries, entry )

      end

   end

   if args.mode == 'function' then
      return entries, vars
   end

end
