local ut      = require('udunits2')
local math    = require('math')

local _M = {}

-- retrieve units system.  if this module isn't being loaded from
-- within raygen (and thus there's no access to the raygen supplied
-- system object), mock something up.
local ok, system = pcall( function () return require('saotrace.raygen.intrinsic.units').system() end )

if not ok then
   local handler = ut.set_error_message_handler( 'ignore' )
   system = require( 'udunits2.system').read_xml()
   ut.set_error_message_handler( handler )
end

-- these should really hook into the C++ side's data rather than
-- generating new unit objects

_M.angular = {
   radian    = system:get_unit_by_name( 'radian' ),
   degree    = system:get_unit_by_name( 'degree' ),
   arcsecond = system:get_unit_by_name( 'arcsecond' ),
   arcminute = system:get_unit_by_name( 'arcminute' ),
}

_M.linear = {
   mm        = system:parse( 'mm', ut.ASCII ),
}

_M.time = {
   seconds   = system:get_unit_by_name( 'seconds' ),
}


-----------------------------------------------------------
local function _normalize_value( arg, input_unit )

   local value

   if type(arg) == 'table' then

      if not arg[1] or not arg[2] then
	 error( "input table argument is not a valid {value, unit} specification", 0 )
      end

      if ut.isa( arg[2], 'unit' ) then

	 input_unit = arg[2]

      elseif type( arg[2] ) == 'string' then

	 local unit = ut.trim( arg[2], ut.ASCII )
	 input_unit = assert( (system:parse( unit, ut.ASCII )),
			      "unknown unit: " .. unit )
      else

	 error( "input table argument is not a valid {value, unit} specification", 0 )

      end

      value = arg[1]

      if type(value) ~= 'number' then
	 error( "input table argument is not a valid {value, unit} specification", 0 )
      end

   else

      -- default is a pure number
      value = tonumber( arg )

      -- not a pure number, parse it and get its units
      if value == nil then

	 arg = ut.trim( arg, ut.ASCII )
	 input_unit = assert( (system:parse( arg, ut.ASCII )),
			      "unknown unit: " .. arg, 0 )
	 value = 1
      end

   end

   return value, input_unit

end

--- normalize an input value
-- convert a value to a { number, unit } pair
-- @param arg the input value it may have one of the following forms
--    * number
--    * "number unit"
--    * { number, "unit" }
-- @param input_unit  a udunits2 unit. use if the input value if it doesn't specify a unit
-- @return number numeric part of value
-- @return unit   udunits2 unit object

function _M.normalize_value( ... )

   -- need to set and reset udunits2 error message handler,
   -- so wrap actual call in a pcall

   -- quiet error messages; handle them ourselves
   local handler = ut.set_error_message_handler( 'ignore' )

   local ok, value, input_unit = pcall( _normalize_value, ... )

   -- reset handler
   ut.set_error_message_handler( handler )

   if not ok then
      error( value )
   end

   return value, input_unit

end

-----------------------------------------------------------
--- convert a value between units.
-- convert a given value to another unit.  use a default unit
-- if not specified.
-- @param arg the input value it may have one of the following forms
--    * number
--    * "number unit"
--    * { number, unit }
-- @param input_unit  a udunits2 unit. use if the input value if it doesn't specify a unit
-- @param output_unit a udunits2 unit. the output unit
-- @return a {value, ut_unit} pair
function _M.convert( arg, input_unit, output_unit, validate )

   local value, input_unit = _M.normalize_value( arg, input_unit )

   if not input_unit then
      error( "no units specified" )
   end

   local cvter = input_unit:get_converter( output_unit )
   if not cvter then
      local name = output_unit:get_name( ut.ASCII )
      name = name or 'unknown'
      error( "can't convert " .. arg .. " to " .. name, 0 )
   end

   value = { cvter:convert(value), output_unit }

   -- something good happened; see if further validation is required
   if validate then

      local ok, value = validate( value )
      if not ok then
	 error( value, 0 )
      end
   end

   return value

end


function _M.convert_angle (arg, input_unit, output_unit, validate )

   -- quiet error messages; handle them ourselves
   local handler = ut.set_error_message_handler( 'ignore' )

   if type(input_unit) == 'string' then
      input_unit = assert( _M.angular[input_unit],
			   "input_unit: unknown unit: " .. input_unit )
   end

   if type(output_unit) == 'string' then
      output_unit = assert( _M.angular[output_unit],
			    "output_unit: unknown unit: " .. output_unit )
   end

   -- capture any errors; message will be in variable value if so
   local ok, value = pcall( _M.convert, arg, input_unit, output_unit, validate )

   -- it's all about this, really
   ut.set_error_message_handler( handler )

   return ok, value

end

--- convert a linear value at a distance z to an angular value in radians
-- the input units for the value parameter must be either linear or angular
-- @param value a list of { number, udunits2 unit }
-- @param z the distance from the observer in units of mm
function _M.cvt_linear_to_angular( extent, z )

   local cvt

   local value = extent[1]
   local unit  = extent[2]

   -- is it angular?
   cvt = unit:get_converter( angular.radian )
   if cvt then
      value = cvt( value )
   else
      cvt = unit:get_converter( angular.mm )
      if not cvt then
	 error( "internal error: input unit is not an angular or linear distance unit" )
      end

      value = math.atan2( cvt( extent ), math.fabs( z ) )

   end

   return value
end


return _M
