/* spline_coef.c */

#define ASCII_ID      "    "
#define NEW_BINARY_ID "BSPL"
#define OLD_BINARY_ID  24
#define FITS_ID       "SIMP"

#include <float.h>            /* DBL_MAX */
#include <math.h>
#include <stdio.h>
#include <string.h>

#include <config.h>

#include <fitsio.h>

#include <BsplineEval/bsplineval.h>
#include <BsplineEval/spline_coef.h>
#include <mathconst/mathconst.h>
#include <tracefct/tracefct.h>
#include <tracefct/exiterrvals.h>

#include <suplib/str.h>

#include "splines.h"
#include "spline_format.h"

#define SPLCOEFFOFFSET( arg ) FUN_OFFSET( spline_coef*, arg )

/*-----------------------------------------------------------------------
 * static data to hold information between calls.
 *
 * The user should free the memory associated
 * with s_scoef and s_bspline by calling SPLINE_CLEANUP_F77( )
 */
static spline_coef   s_scoef;
static BsplineInput* s_bspline = NULL;

static char s_spline_version_string[8] = "BSPL0001";

/*=======================================================================
 * allocate/deallocate
 */
/*-----------------------------------------------------------------------*/
void
alloc_spline_coefs(void)
{
  alloc_spline_data();
}

/*-----------------------------------------------------------------------*/
void
free_spline_coefs(void)
{
  free_spline_data();
}

/*-----------------------------------------------------------------------*/
void 
alloc_spline_data(void)
{
  tf_enter( "alloc_spline_data" );

  s_scoef.z_knots = (double*) malloc( s_scoef.qz * sizeof(double) );
  if (! s_scoef.z_knots )
    tf_exit( ExitERR_alloc, "unable to allocate space for z knots" );

  s_scoef.t_knots = (double*) 
                      malloc( s_scoef.num_theta_knots * sizeof(double) );
  if (! s_scoef.t_knots )
    tf_exit( ExitERR_alloc, "unable to allocate space for theta knots" );

  s_scoef.c = (double*) malloc( (s_scoef.qz * s_scoef.num_theta_knots) 
                             * sizeof(double) );
  if (! s_scoef.c )
    tf_exit( ExitERR_alloc, "unable to allocate space for spline amplitude" );

  s_scoef.cdifz = (double*) malloc( ((s_scoef.qz-1) * s_scoef.num_theta_knots)
				    * sizeof(double) );
  if (! s_scoef.cdifz )
    tf_exit( ExitERR_alloc, "unable to allocate space for dq/dz" );

  s_scoef.cdift = (double*) malloc( (s_scoef.qz * s_scoef.num_theta_knots)
                                 * sizeof(double) );
  if (! s_scoef.cdift )
    tf_exit( ExitERR_alloc, "unable to allocate space for dq/dtheta" );

  tf_leave();
}

/*-----------------------------------------------------------------------*/
void free_spline_data(void) {
  if ( NULL != s_scoef.c       ) { free( s_scoef.c       ); }
  if ( NULL != s_scoef.cdifz   ) { free( s_scoef.cdifz   ); }
  if ( NULL != s_scoef.cdift   ) { free( s_scoef.cdift   ); }
  if ( NULL != s_scoef.t_knots ) { free( s_scoef.t_knots ); }
  if ( NULL != s_scoef.z_knots ) { free( s_scoef.z_knots ); }
}

/*-----------------------------------------------------------------------*/
void 
SPLINE_CLEANUP_F77( void )
{
  tf_enter( "spline_cleanup_" );
  free_spline_coefs();
  if ( NULL != s_bspline )
    Bspline_free( s_bspline );
  tf_leave();
}

/*=======================================================================
 * mutators: get/set
 */

/*-----------------------------------------------------------------------*/
spline_coef*
get_spline_coef( void )
{
  return &s_scoef;
}

/*-----------------------------------------------------------------------*/
void get_spline_hdr( int* z_order, int* theta_order, int* qz,
		     int* num_z_knots, int* num_theta_knots ) {
  *z_order         = s_scoef.z_order;
  *theta_order     = s_scoef.theta_order;
  *qz              = s_scoef.qz;
  *num_z_knots     = s_scoef.num_z_knots;
  *num_theta_knots = s_scoef.num_theta_knots;
}

/*-----------------------------------------------------------------------*/
void set_spline_hdr( int z_order, int theta_order,
		     int qz, int num_theta_knots ) {
  s_scoef.z_order         = z_order;
  s_scoef.theta_order     = theta_order;
  s_scoef.qz              = qz;
  s_scoef.num_z_knots     = qz - z_order;
  s_scoef.num_theta_knots = num_theta_knots;
}

/*-----------------------------------------------------------------------*/
void get_spline_limits( double* zmin, double* zmax,
                   double* tmin, double* tmax ) {
  *zmin = s_scoef.zmin;
  *zmax = s_scoef.zmax;
  *tmin = s_scoef.tmin;
  *tmax = s_scoef.tmax;
}

/*-----------------------------------------------------------------------*/
void set_spline_limits( double zmin, double zmax,
			double tmin, double tmax ) {
  s_scoef.zmin = zmin;
  s_scoef.zmax = zmax;
  s_scoef.tmin = tmin;
  s_scoef.tmax = tmax;
}

/*-----------------------------------------------------------------------*/
void get_spline_dataptrs( double** z_knots, double** t_knots,
			  double** c, double** cdifz, double** cdift ) {
  *z_knots = s_scoef.z_knots;
  *t_knots = s_scoef.t_knots;
  *c       = s_scoef.c;
  *cdifz   = s_scoef.cdifz;
  *cdift   = s_scoef.cdift;
}

/*-----------------------------------------------------------------------*/
double eval_rms_amplitude(void) {
  int ntot   = 0;
  int jt;
  int jtmax  = s_scoef.num_theta_knots;
  int jz;
  int jzmax  = s_scoef.num_z_knots;
  double sum = 0.0;
  double theta;
  double zn;
  double h;

  tf_enter( "eval_rms_amplitude" );

  for ( jt = 0 ; jt < jtmax ; ++jt )
  {
    theta = s_scoef.t_knots[jt];
    if ( theta < 0.0        ) { theta += M_2_PI; }
    if ( theta > s_scoef.tmax ) { theta = fmod( theta, M_2_PI ); }

    for ( jz = 0 ; jz < jzmax ; ++jz )
    {
      zn = s_scoef.z_knots[jz];

      if ( Bspline_Success !=
           Bspline_eval( theta, zn, s_bspline, &h ) )
      {
        fprintf( stderr,
                 "eval_rms_amplitude: Bspline_eval returned with "
                 "an error  t=%.14f, z=%.14f\n", theta, zn );
        return -DBL_MAX;
      }
      ++ntot;
      sum += h * h;
    }
  }
  tf_leave( );
  return sqrt(sum / ntot);
}

/*=======================================================================
 * evaluate the spline
 */
/*-----------------------------------------------------------------------*/
int
SPLINE_EVAL_F77( double const* theta, double const* zn, 
                 double* s, double* dsdz, double* dsdt  )
{
  BsplineResult result;
  double        theta_tmp = *theta;
  BsplineErr    err;

  if ( theta_tmp < 0.0          ) { theta_tmp += M_2_PI; }
  if ( theta_tmp > s_scoef.tmax ) { theta_tmp = fmod( theta_tmp, M_2_PI ); }

  err = Bspline_eval_derivs( theta_tmp, *zn, s_bspline, &result );
  if ( Bspline_Success != err )
  {
    fprintf( stderr, 
             "spline_eval: Bspline_eval_derivs returned with an error t=%.14f, "
             "z=%.14f\n", theta_tmp, *zn );
    fprintf( stderr, "spline_eval_: err is %d, theta_tmp = %e zn = %e\n",
             err, theta_tmp, *zn );
    Bspline_err_msg( err, stderr );
    return EXIT_FAILURE;
  }

  *s    = result.s;
  *dsdz = result.dsdz;
  *dsdt = result.dsdt;

  return EXIT_SUCCESS;

}

/*=======================================================================
 * read in the spline data
 */
/*-----------------------------------------------------------------------*/
void
READ_SPLINE_DATA_F77( char const filename[], int const* io_unit, 
                      double* rms_amplitude, mst_fortran_charlen_t ln )
{
  int     io_dir    = 0;
  int     iostat    = 0;
  int     io_format = 0;
  int	  eval_amplitude = 0;
  int     qzm1;
  int     test_file;
  double* scr;
  FILE*   stream;

  tf_enter( "read_spline_data_" );

  /*
   * open stream for reading; determine the spline file format...
   */
  if ( (stream = fopen(filename, "r")) == NULL )
    tf_exit( ExitERR_fopen, "unable to open file %s\n", filename );

  if ( fread( &test_file, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read beginning of file" );

  if (! strncmp( (char*)&test_file, FITS_ID, (size_t) 4 ) )
    io_format = (int)EFits;

  else if ( test_file == OLD_BINARY_ID )
    io_format = (int)EOldBinary;

  else if (! strncmp( (char*)&test_file, NEW_BINARY_ID, (size_t) 4 ) )
    io_format = (int)ENewBinary;

  else if (! strncmp( (char*)&test_file, ASCII_ID, (size_t) 4 ) )
    io_format = (int)EAscii;

  else
    tf_exit( ExitERR_IOerr, "Unrecognized spline file format" );

  fclose(stream);

  /* FITS format */
  switch( io_format )
  {
  case EFits:
    read_fits_spline_data( filename );
    break;

  /* New Binary format */
  case ENewBinary:
    cread_spline_data( filename );
    break;

  /* Old Binary format or Ascii */
  case EAscii:
  case EOldBinary:
    {
      /* interface size_t to int */
      int     z_order = s_scoef.z_order;
      int          qz = s_scoef.qz;
      int theta_order = s_scoef.theta_order;
      int num_theta_knots = s_scoef.num_theta_knots;
      int     ier = 0;

      /* need to evalute RMS amplitude, as it's not stored in file */ 
      eval_amplitude = 1;
    
      ln = strlen( filename );


      OPEN_FTN_IO_F77( filename, io_unit, &io_format, &io_dir, &iostat, ln );

      READ_FTN_HDR_F77( io_unit, &io_format, 
			&z_order, &qz, 
			&theta_order, &num_theta_knots,
			&s_scoef.zmin, &s_scoef.zmax, 
			&s_scoef.tmin, &s_scoef.tmax, &ier );

      if ( ier )
	tf_die( "error reading header for spline file %s\n", filename );

      /* reverse int to size_t */
      s_scoef.z_order = z_order;
      s_scoef.qz      = qz;
      s_scoef.theta_order = theta_order;
      s_scoef.num_theta_knots = num_theta_knots;

      s_scoef.num_z_knots = s_scoef.qz - s_scoef.z_order;

      qzm1 = s_scoef.qz - 1;

      alloc_spline_coefs();

      scr = (double*) 
	malloc( s_scoef.qz * s_scoef.num_theta_knots * sizeof(double) ); 

      if (! scr )
	tf_exit( ExitERR_alloc, "unable to allocate space for scratch array" );


      READ_FTN_SPLDATA_F77( io_unit, &io_format, 
			    s_scoef.z_knots, s_scoef.t_knots, 
			    s_scoef.c, s_scoef.cdifz, s_scoef.cdift, scr, 
			    &qz, &qzm1, &num_theta_knots );

      free(scr);

      CLOSE_FTN_IO_F77( io_unit );
    }

  }


  /* Allocate the memory used by the BsplineEval library */
  s_bspline = (BsplineInput*) Bspline_init( &s_scoef );
  if ( NULL == s_bspline ) 
      tf_exit( ExitERR_alloc, "unable to allocate space for eval_spline\n" );


  *rms_amplitude =
      eval_amplitude
    ? eval_rms_amplitude()
    : s_scoef.c_rms_amplitude;

}

static void
fits_dump_errmsgs( int status ) {

    char msg[81];

    while( fits_read_errmsg( msg ) )
	tf_message( "%s\n", msg );

    fits_get_errstatus( status, msg );
    tf_message( "%s\n", msg );

}

/*-----------------------------------------------------------------------
 * read_fits_spline_hdrs: static helper function
 */
static void
read_fits_spline_hdrs( fitsfile *fptr )
{
  int status = 0;

  /* use to convert from cfitsio TLONG to size_t in s_scoef struct */
  long tlong;

  tf_enter( "read_fits_spline_hdrs" );

  fits_read_key( fptr, TDOUBLE, "RMSAMP",  &s_scoef.c_rms_amplitude, NULL, &status );
  fits_read_key( fptr, TDOUBLE, "ZMIN",    &s_scoef.zmin           , NULL, &status );
  fits_read_key( fptr, TDOUBLE, "ZMAX",    &s_scoef.zmax           , NULL, &status );
  fits_read_key( fptr, TDOUBLE, "TMIN",    &s_scoef.tmin           , NULL, &status );
  fits_read_key( fptr, TDOUBLE, "TMAX",    &s_scoef.tmax           , NULL, &status );

  fits_read_key( fptr, TLONG,   "ZORDER",  &tlong, NULL, &status );
  s_scoef.z_order = tlong;

  fits_read_key( fptr, TLONG,   "TORDER",  &tlong, NULL, &status );
  s_scoef.theta_order = tlong;

  fits_read_key( fptr, TLONG,   "NZKNOTS", &tlong, NULL, &status );
  s_scoef.num_z_knots = tlong;

  fits_read_key( fptr, TLONG,   "NTKNOTS", &tlong, NULL, &status );
  s_scoef.num_theta_knots = tlong;

  if ( status ) {
      fits_dump_errmsgs( status );
      tf_die( "error reading spline headers" );
  }

  s_scoef.qz = s_scoef.num_z_knots + s_scoef.z_order;

  tf_leave();
}

/*-----------------------------------------------------------------------*/
static
int read_fits_column( fitsfile *fptr, const char* colname, double* data ) {

  int status = 0;
  int colnum;
#undef MAX_DIM
#define MAX_DIM 2
  int naxis;
  long naxes[MAX_DIM];
  long nrows;
  long nelements;
  char *_colname = str_dup( colname );
  int i;

  fits_get_colnum( fptr, CASEINSEN, _colname, &colnum, &status );

  free( _colname );

  fits_read_tdim( fptr, colnum, MAX_DIM, &naxis, naxes, &status );
  fits_get_num_rows( fptr, &nrows, &status );

  nelements = nrows;

  for ( i = 0 ; i < naxis ; i++ )
      nelements *= naxes[i];

  fits_read_col( fptr, TDOUBLE, colnum, (LONGLONG) 1, (LONGLONG) 1, nelements,
		 NULL, data, NULL, &status );

  return status;
}


/*-----------------------------------------------------------------------*/
void
read_fits_spline_data( const char filename[] )
{
#undef MAX_DIM
#define MAX_DIM 1

  int status = 0;
  fitsfile *fptr;

  tf_enter( "read_fits_spline_data" );

  if ( fits_open_file( &fptr, filename, READONLY, &status ) ) {
      fits_dump_errmsgs( status );
      tf_die( "error opening %s", filename );
  }

  /* ------------- */


  if ( fits_movnam_hdu( fptr, BINARY_TBL, "SPLKNOTS", 0, &status ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: unable to find SPLKNOTS HDU", filename );
  }

  read_fits_spline_hdrs( fptr );

  alloc_spline_data( );

  if ( read_fits_column( fptr, "Z", s_scoef.z_knots ) )  {
      fits_dump_errmsgs( status );
      tf_die( "%s: error reading column Z in HDU SPLKNOTS", filename );
  }

  if ( read_fits_column( fptr, "THETA", s_scoef.t_knots ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: error reading column THETA in HDU SPLKNOTS", filename );
  }

  /* ------------- */

  if ( fits_movnam_hdu( fptr, BINARY_TBL, "SPLCOEFF", 0, &status ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: unable to find SPLCOEFF HDU", filename );
  }

  if ( read_fits_column( fptr, "COEF", s_scoef.c ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: error reading column COEF in HDU SPLCOEFF", filename );
  }

  if ( read_fits_column( fptr, "COEFDIFZ", s_scoef.cdifz ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: error reading column COEFDIFZ in HDU SPLCOEFF", filename );
  }

  if ( read_fits_column( fptr, "COEFDIFT", s_scoef.cdift ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: error reading column COEFDIFT in HDU SPLCOEFF", filename );
  }

  if ( fits_close_file( fptr, &status ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: error closing" );
  }

  tf_leave();

  /*
   * The user should free the memory associated
   * with s_scoef by calling SPLINE_CLEANUP_F77( )
   */
}

/*-----------------------------------------------------------------------*/
void
cread_spline_data( char const filename[] )
{
  size_t     jt;
  size_t     num_rows;
  double *c_data;
  double *cz_data;
  double *ct_data;
  FILE   *stream;
  char    spline_version_string[8] = "BSPL0001";

  tf_enter( "cread_spline_data" );

  /*
   * open stream for reading
   */
  if ( (stream = fopen(filename, "r")) == NULL )
    tf_exit( ExitERR_fopen, "unable to open file %s\n", filename );

  /*
   * read in header stuff
   */
  if ( fread( spline_version_string, 
              sizeof(spline_version_string), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read file spline version string" );

  if ( fread( &s_scoef.c_rms_amplitude, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read rms_amplitude" );

  if ( fread( &s_scoef.z_order, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read z_order" );

  if ( fread( &s_scoef.theta_order, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read theta_order" );

  if ( fread( &s_scoef.num_theta_knots, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read num_theta_knots" );

  if ( fread( &s_scoef.qz, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read qz" );

  if ( fread( &s_scoef.z_order, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read z_order" );

  s_scoef.num_z_knots = s_scoef.qz - s_scoef.z_order;

  /*
   * read in spline limits 
   */
  if ( fread( &s_scoef.zmin, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read zmin" );

  if ( fread( &s_scoef.zmax, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read zmax" );

  if ( fread( &s_scoef.tmin, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read tmin" );

  if ( fread( &s_scoef.tmax, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read tmax" );


  /*
   * allocate space to hold the data...
   */
  alloc_spline_coefs();

  /*
   * read in spline knots
   */
  if ( fread( s_scoef.z_knots, s_scoef.qz * sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read z knots" );

  if ( fread( s_scoef.t_knots, s_scoef.num_theta_knots * sizeof(double), (size_t) 1,
	      stream ) != 1 )
    tf_exit( ExitERR_fread, "unable to read theta knots" );


  /*
   * read in spline data
   */
  c_data   = s_scoef.c;
  cz_data  = s_scoef.cdifz;
  ct_data  = s_scoef.cdift;
  num_rows = s_scoef.qz;
  for ( jt = 0 ; jt < s_scoef.num_theta_knots ; ++jt )
  {
    if ( fread( c_data, num_rows * sizeof(double), (size_t) 1, stream ) != 1 )
      tf_exit( ExitERR_fread, "unable to read c col %lu", jt );

    if ( fread( cz_data, (num_rows-1) * sizeof(double), (size_t) 1, stream ) != 1 )
      tf_exit( ExitERR_fread, "unable to read cdifz col %lu", jt );

    if ( fread( ct_data, num_rows * sizeof(double), (size_t) 1, stream ) != 1 )
      tf_exit( ExitERR_fread, "unable to read cdift col %lu", jt );

    c_data  += num_rows;
    cz_data += num_rows - 1;
    ct_data += num_rows;
  }

  fclose( stream );

  tf_leave();
}

/*=======================================================================
 * write out the spline data
 */
/*-----------------------------------------------------------------------*/
void
WRITE_SPLINE_DATA_F77( char const filename[], int const* io_unit, 
                       int const* io_format, mst_fortran_charlen_t ln )
{
  int     io_dir = 1;
  int     iostat = 0;
  int     qzm1;
  int     ier;

  tf_enter( "write_spline_data_" );

  if ( NULL == s_bspline )
    s_bspline = (BsplineInput*) Bspline_init( &s_scoef );

  switch (*io_format)
  {
  case EAscii:
  case EOldBinary:
    {
      double *scr;

      /* interface size_t to int */
      int     z_order = s_scoef.z_order;
      int          qz = s_scoef.qz;
      int theta_order = s_scoef.theta_order;
      int num_theta_knots = s_scoef.num_theta_knots;

      ln = strlen( filename );
      OPEN_FTN_IO_F77( filename, io_unit, io_format, &io_dir, &iostat, ln );

      qzm1 = s_scoef.qz - 1;


      scr = (double*) 
	malloc( s_scoef.qz * s_scoef.num_theta_knots * sizeof(double) );
      if (! scr )
	tf_exit( ExitERR_alloc, "unable to allocate space for scratch array" );


      WRITE_FTN_HDR_F77( io_unit, io_format,
			 &z_order, &qz, 
			 &theta_order, &num_theta_knots, 
			 &s_scoef.zmin, &s_scoef.zmax, 
			 &s_scoef.tmin, &s_scoef.tmax, &ier );

      WRITE_FTN_SPLDATA_F77( io_unit, io_format, s_scoef.z_knots,
			     s_scoef.t_knots, s_scoef.c, s_scoef.cdifz,
			     s_scoef.cdift, scr, &qz, &qzm1,
			     &num_theta_knots );

      free(scr);

      CLOSE_FTN_IO_F77( io_unit );
    }
    break;

  case ENewBinary:
    s_scoef.c_rms_amplitude = eval_rms_amplitude();
    cwrite_spline_data( filename );
    break;

  case ERDB_DFM:
    s_scoef.c_rms_amplitude = eval_rms_amplitude();
    write_rdb( filename );
    break;

  case EFits:
    s_scoef.c_rms_amplitude = eval_rms_amplitude();
    write_fits_spline_data( filename );
    break;
  }

  tf_leave( );
}

/*-----------------------------------------------------------------------*/
void
cwrite_spline_data( char const filename[] )
{
  size_t  jt;
  int     num_rows;
  double *c_data;
  double *cz_data;
  double *ct_data;
  FILE   *stream;

  tf_enter( "write_spline_data" );

  /*
   * open stream for writing
   */
  if ( (stream = fopen(filename, "w")) == NULL )
    tf_exit( ExitERR_fopen, "unable to open file %s\n", filename );

  /*
   * write out header stuff
   */
  if ( fwrite( s_spline_version_string, 
               sizeof(s_spline_version_string), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write spline version string" );

  if ( fwrite( &s_scoef.c_rms_amplitude, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write rms_amplitude" );

  if ( fwrite( &s_scoef.z_order, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write z_order" );

  if ( fwrite( &s_scoef.theta_order, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write theta_order" );

  if ( fwrite( &s_scoef.num_theta_knots, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write num_theta_knots" );

  if ( fwrite( &s_scoef.qz, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write qz" );

  if ( fwrite( &s_scoef.z_order, sizeof(int), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write z_order" );


  /*
   * write out spline limits
   */
  if ( fwrite( &s_scoef.zmin, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write zmin" );

  if ( fwrite( &s_scoef.zmax, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write zmax" );

  if ( fwrite( &s_scoef.tmin, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write tmin" );

  if ( fwrite( &s_scoef.tmax, sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write tmax" );


  /*
   * write out spline knots
   */
  if ( fwrite( s_scoef.z_knots, s_scoef.qz * sizeof(double), (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write z knots" );

  if ( fwrite( s_scoef.t_knots, s_scoef.num_theta_knots * sizeof(double),
	       (size_t) 1, stream ) != 1 )
    tf_exit( ExitERR_fwrite, "unable to write theta knots" );


  /*
   * write in spline data
   */
  c_data   = s_scoef.c;
  cz_data  = s_scoef.cdifz;
  ct_data  = s_scoef.cdift;
  num_rows = s_scoef.qz;
  for ( jt = 0 ; jt < s_scoef.num_theta_knots ; ++jt )
  {
    if ( fwrite( c_data, num_rows * sizeof(double), (size_t) 1, stream ) != 1 )
      tf_exit( ExitERR_fwrite, "unable to write c col %d", jt );

    if ( fwrite( cz_data, (num_rows-1) * sizeof(double), (size_t) 1, stream ) != 1 )
      tf_exit( ExitERR_fwrite, "unable to write cdifz col %d", jt );

    if ( fwrite( ct_data, num_rows * sizeof(double), (size_t) 1, stream ) != 1 )
      tf_exit( ExitERR_fwrite, "unable to write cdift col %d", jt );

    c_data  += num_rows;
    cz_data += num_rows - 1;
    ct_data += num_rows;
  }

  fclose( stream );

  tf_leave();
}

/*-----------------------------------------------------------------------
 * write_fits_spline_hdrs: static helper function
 */
static void
write_fits_spline_hdrs( fitsfile *fptr )
{
  double splvers=1.0;

  int status = 0;
  unsigned long tlong;

  tf_enter( "write_fits_spline_hdrs" );

  fits_write_key( fptr, TDOUBLE, "SPLVERS", &splvers, "Version of the Spline Format", &status );
  fits_write_key( fptr, TDOUBLE, "RMSAMP",  &s_scoef.c_rms_amplitude, "RMS Amplitude as derived by spline_convert_format", &status );

  tlong = s_scoef.z_order;
  fits_write_key( fptr, TLONG, "ZORDER", &tlong, "Order of Z", &status );

  tlong = s_scoef.theta_order;
  fits_write_key( fptr, TLONG, "TORDER", &tlong, "Order of theta", &status );

  tlong = s_scoef.num_z_knots;
  fits_write_key( fptr, TLONG, "NZKNOTS", &tlong, "Number of unaugmented knots in Z", &status );

  tlong = s_scoef.num_theta_knots;
  fits_write_key( fptr, TLONG, "NTKNOTS", &tlong, "Number of unaugmented knots in THETA", &status );

  fits_write_key( fptr, TDOUBLE, "ZMIN", &s_scoef.zmin, "Lower bound of valid range of Z", &status );
  fits_write_key( fptr, TDOUBLE, "ZMAX", &s_scoef.zmax, "Upper bound of valid range of Z", &status );

  fits_write_key( fptr, TDOUBLE, "TMIN", &s_scoef.tmin, "Lower bound of valid range of THETA", &status );
  fits_write_key( fptr, TDOUBLE, "TMAX", &s_scoef.tmax, "Upper bound of valid range of THETA", &status );


  if ( status) {
    fits_dump_errmsgs( status );
    tf_die( "error writing FITS spline header" );
  }

  tf_leave();
}

/*-----------------------------------------------------------------------*/

static
int write_fits_column( fitsfile *fptr,
		       const char* colname,
		       int naxis,
		       long* naxes,
		       long nrows,
		       double* data ) {

  int status = 0;
  int colnum;
  long nelements = nrows;
  int i;
  char *_colname = str_dup( colname );


  fits_get_colnum( fptr, CASEINSEN, _colname, &colnum, &status );
  free( _colname );

  for ( i = 0 ; i < naxis ; i++ )
      nelements *= naxes[i];

  fits_write_tdim( fptr, colnum, naxis, naxes, &status );

  if ( status) {
    fits_dump_errmsgs( status );
    tf_die( "error writing FITS column" );
  }

  fits_write_col( fptr, TDOUBLE, colnum, (LONGLONG) 1, (LONGLONG) 1,
		  nelements, data, &status );

  return status;
}

/*-----------------------------------------------------------------------*/
void
write_fits_spline_data( char const filename[] )
{
  fitsfile *fptr;
  char *nfile;
  int status = 0;

  char *ttype[3];
  char tformv[3][FLEN_VALUE];
  char *tform[3];
  long naxes[1];

  tform[0] = tformv[0];
  tform[1] = tformv[1];
  tform[2] = tformv[2];

  tf_enter( "write_fits_spline_data" );

  nfile = (char*) malloc( strlen(filename) + 2  );
  if ( NULL == nfile )
      tf_die( "error allocating space for filename" );

  nfile[0] = '!';
  strcpy( nfile+1, filename );

  if ( fits_create_file( &fptr, nfile, &status ) )
    tf_die( "unable to create %s\n", filename );

  free( nfile );

  {
      ttype[0] = "Z";
      ttype[1] = "THETA";

      sprintf( tform[0], "%ldD", s_scoef.qz );
      sprintf( tform[1], "%ldD", s_scoef.num_theta_knots );

      if ( fits_create_tbl( fptr, BINARY_TBL, (LONGLONG) 0, 2, ttype, tform, NULL, "SPLKNOTS", &status ) )  {
	  fits_dump_errmsgs( status );
	  tf_die( "error creating SPLKNOTS extension" );
      }

      write_fits_spline_hdrs( fptr );

      naxes[0] = s_scoef.qz;
      if ( write_fits_column( fptr, "Z", 1, naxes, (LONGLONG) 1, s_scoef.z_knots ) ) {
	  fits_dump_errmsgs( status );
	  tf_die( "%s: error writing Z column in SPLKNOTS extension", filename );
      }

      naxes[0] = s_scoef.num_theta_knots;
      if ( write_fits_column( fptr, "THETA", 1, naxes, (LONGLONG)1, s_scoef.t_knots ) ) {
	  fits_dump_errmsgs( status );
	  tf_die( "%s: error writing THETA column in SPLKNOTS extension", filename );
      }

  }

  {
      ttype[0] = "COEF";
      ttype[1] = "COEFDIFZ";
      ttype[2] = "COEFDIFT";

      sprintf( tform[0], "%ldD", s_scoef.qz );
      sprintf( tform[1], "%ldD", s_scoef.qz-1 );
      sprintf( tform[2], "%ldD", s_scoef.qz );

      if ( fits_create_tbl( fptr, BINARY_TBL, (LONGLONG) 0, 3,
			    ttype, tform, NULL, "SPLCOEFF", &status ) ) {
	  fits_dump_errmsgs( status );
	  tf_die( "error creating SPLCOEFF extension" );
      }

      naxes[0] = s_scoef.qz;
      if ( write_fits_column( fptr, "COEF", 1, naxes,
			      (LONGLONG) s_scoef.num_theta_knots,
			      s_scoef.c ) ) {
	  fits_dump_errmsgs( status );
	  tf_die( "%s: error writing COEF column in SPLCOEFF extension",
		  filename );
      }

      naxes[0] = s_scoef.qz - 1;
      if ( write_fits_column( fptr, "COEFDIFZ", 1, naxes,
			      (LONGLONG) s_scoef.num_theta_knots,
			      s_scoef.cdifz ) ) {
	  fits_dump_errmsgs( status );
	  tf_die( "%s: error writing COEFDIFZ column in SPLCOEFF extension",
		  filename );
      }

      naxes[0] = s_scoef.qz;
      if ( write_fits_column( fptr, "COEFDIFT", 1, naxes,
			      (LONGLONG) s_scoef.num_theta_knots,
			      s_scoef.cdift ) ) {
	  fits_dump_errmsgs( status );
	  tf_die( "%s: error writing COEFDIFT column in SPLCOEFF extension",
		  filename );
      }

  }

  if ( fits_close_file( fptr, &status ) ) {
      fits_dump_errmsgs( status );
      tf_die( "%s: error closing" );
  }

  tf_leave();

}

/*-----------------------------------------------------------------------*/
void
write_rdb( char const filename[] )
{
  size_t jt;
  size_t jtmax  = s_scoef.num_theta_knots;
  size_t jz;
  size_t jzmax  = s_scoef.num_z_knots;
  double theta;
  double zn;
  double h;
  FILE   *stream;

  tf_enter( "write_rdb" );

  /*
   * open stream for writing
   */
  if ( (stream = fopen(filename, "w")) == NULL )
    tf_exit( ExitERR_fopen, "unable to open file %s\n", filename );

  /*
   * write out header stuff
   */
  fprintf( stream, "# spline deformation amplitudes\n" );
  fprintf( stream, "# z_order         %lu\n", s_scoef.z_order );
  fprintf( stream, "# theta_order     %lu\n", s_scoef.theta_order );
  fprintf( stream, "# qz              %lu\n", s_scoef.qz );
  fprintf( stream, "# num_z_knots     %lu\n", s_scoef.num_z_knots );
  fprintf( stream, "# num_theta_knots %lu\n", s_scoef.num_theta_knots );
  fprintf( stream, "# zmin zmax       %e %e\n", s_scoef.zmin, s_scoef.zmax );
  fprintf( stream, "# tmin tmax       %e %e\n", s_scoef.tmin, s_scoef.tmax );
  fprintf( stream, "# rms amplitude   %e\n", s_scoef.c_rms_amplitude );
  fprintf( stream, "# \n" );
  fprintf( stream, "jtheta\tjz\ttheta\tz\tdfm\n" );
  fprintf( stream, "N\tN\tN\tN\tN\n" );

  for ( jt = 0 ; jt < jtmax ; ++jt )
  {
    theta = s_scoef.t_knots[jt];
    if ( theta < 0.0        ) { theta += M_2_PI; }
    if ( theta > s_scoef.tmax ) { theta = fmod( theta, M_2_PI ); }

    for ( jz = 0 ; jz < jzmax ; ++jz )
    {
      zn = s_scoef.z_knots[jz];

      if ( Bspline_Success !=
           Bspline_eval( theta, zn, s_bspline, &h ) )
      {
        fprintf( stderr,
                 "write_rdb: Bspline_eval returned with "
                 "an error  t=%.14f, z=%.14f\n", theta, zn );
        tf_exit( ExitERR_misc, "Bspline_eval error\n" );
      }

      fprintf( stream, "%lu\t%lu\t%e\t%e\t%e\n", jt, jz, theta, zn, h );
    }
  }
  fclose( stream );

  tf_leave();
}

/*-----------------------------------------------------------------------*/
void
dump_s_scoef()
{
  fprintf(stderr, "DUMP z_order         >%lu<\n", s_scoef.z_order);
  fprintf(stderr, "DUMP theta_order     >%lu<\n", s_scoef.theta_order);
  fprintf(stderr, "DUMP num_z_knots     >%lu<\n", s_scoef.num_z_knots);
  fprintf(stderr, "DUMP num_theta_knots >%lu<\n", s_scoef.num_theta_knots);
  fprintf(stderr, "DUMP qz              >%lu<\n", s_scoef.qz);
  fprintf(stderr, "DUMP zmin            >%e<\n", s_scoef.zmin);
  fprintf(stderr, "DUMP zmax            >%e<\n", s_scoef.zmax);
  fprintf(stderr, "DUMP tmin            >%e<\n", s_scoef.tmin);
  fprintf(stderr, "DUMP tmax            >%e<\n", s_scoef.tmax);
  fprintf(stderr, "DUMP z_knots[0]      >%e<\n", s_scoef.z_knots[0]);
  fprintf(stderr, "DUMP t_knots[0]      >%e<\n", s_scoef.t_knots[0]);
  fprintf(stderr, "DUMP c[0]            >%e<\n", s_scoef.c[0]);
  fprintf(stderr, "DUMP cdifz[0]        >%e<\n", s_scoef.cdifz[0]);
  fprintf(stderr, "DUMP cdift[0]        >%e<\n", s_scoef.cdift[0]);
  fprintf(stderr, "DUMP c_rms_amplitude >%e<\n", s_scoef.c_rms_amplitude);
}
/*

  The code here has a lot of issues!

   1) The variable s_scoef (struct spline_coef) is declared as 
      global static variable, a fortran legacy (I think).  The
      function read_fits_spline_data returns a pointer to the 
      s_scoef. So when the user wants to free the memory within
      s_scoef, it has to call SPLINE_CLEANUP_F77( ) with no arguments

   2) When the function READ_SPLINE_DATA_F77 is called, memory is
      allocated for the global static s_scoef and s_bspline.  So 
      the memory should be free when exiting the program, however
      there really isn't anything that reminds the user that the
      memory should be free.  Ugly.
}

 */
