-- Use Chandra aspect solutions file for driving telescope jitter.

local tonumber = tonumber
local assert = assert

local pi = require('math').pi

local tables = require( 'saotrace.suplib.tables' )

local raygen = require( 'saotrace.raygen' )
local jitter = require( 'saotrace.raygen.jitter' )
local fits   = require( 'saotrace.raygen.fits' )

local vobj = require( 'saotrace.raygen.validate').vobj:new()

local dither_args = {
   file = { type = 'string' },
   ra   = { type = 'angle_ra' },
   dec  = { type = 'angle_dec' },
   roll = { type = 'angle_degree', default = 0 },
   format = tables.merge( nil, jitter.vspec.file.format, { default = 'fits' }),
   extrapolate_max_dt = tables.copy( jitter.vspec.file.extrapolate_max_dt ),

   -- allow selection of either the quaternions or the celestial positions
   -- in the aspect solutions file
   adata = { enum = { 'quat', 'celestial' }, default = 'quat' },
}


local function dither_asol_common( args )

   local ra = args.ra
   local dec = args.dec
   local roll = args.roll

   args.adata = nil
   args.roll = nil
   args.dec = nil
   args.ra = nil

   args.optical_axis = {
      {  angle = roll, axis = {  1,  0,  0 } },
      {  angle = dec,  axis = {  0, -1,  0 } },
      {  angle = ra,   axis = {  0,  0,  1 } },
   }


   -- transformations applied to input coords to get raytrace coords
   args.coord_xfrm = {
      x = { 0,  -1,  0 },
      y = { 0,   0, -1 },
      z = { -1,  0,  0 }
   }


   jitter.file( args )
   jitter.roll( roll )

end

local function dither_asol_chandra( ... )

   -- Tom Aldcroft says that we can extrapolate up to 1/8 of a second
   -- outside of the dither table.

   -- But, it looks like Chandra asol files contain dither bins with
   -- time stamp less than TSTOP so the last time stamp may be almost
   -- one bin width away from TSTOP.
   dither_args.extrapolate_max_dt.default = 0.256251

   local ok, args = vobj:validate( dither_args,  ... )
   assert( ok, args )

   args.name = 'chandra asol'

   if  args.adata == 'quat' then

      args.quat_col = 'q_att'

   else

      args.axes = {
	 { col = 'ra',   axis = { 0,  0, 1 } },
	 { col = 'dec',  axis = { 0, -1, 0 } },
	 { col = 'roll', axis = { 1,  0, 0 } },
      }

   end

   dither_asol_common( args )

end

local function dither_asol_marx( ... )

   dither_args.extrapolate_max_dt.default = 0

   local ok, args = vobj:validate( dither_args,  ... )
   assert( ok, args )

   args.name = 'marx asol'

   local table = fits.table:new( args.file );

   if args.adata == 'quat' then

      if not table:exists_column( 'q_att' ) then
         error( "quaternion data requested but column 'q_att' doesn't exist in " .. args.file )
      end

      args.quat_col = 'q_att'

   else

      args.axes = {
	 { col = 'ra',   axis = { 0,  0, 1 } },
	 { col = 'dec',  axis = { 0, -1, 0 } },
	 { col = 'roll', axis = { 1,  0, 0 } },
      }

   end

   dither_asol_common( args )

end

local function lissajous( args )

   jitter.lissajous( 'lj', args )

end


return { dither_asol = dither_asol_chandra,
	 dither_asol_chandra = dither_asol_chandra,
	 dither_asol_marx = dither_asol_marx,
	 lissajous = lissajous,
	 roll = jitter.roll
      }
