local package = require( 'package' )
local string  = require('string')

local setup = require 'setup'

local intrinsic = require( 'saotrace.raygen.mock_intrinsics' )
local raygen = require( 'saotrace.raygen' )
local coords = require('saotrace.raygen.coords')
local units  = require('saotrace.raygen.units')

local chandra_coords = require('chandra.saotrace.coords')

local srcdir = os.getenv('srcdir') or '.'
local ea_db = table.concat( { srcdir, 'tests', 'data', 'annulus.rdb'}, '/' )

local convert = units.convert
local argfmt = string.format( '%%s; file = %q; shell = %d', ea_db, 1 )

local min2rad = coords.min2rad
local deg2rad = coords.deg2rad

describe( "focus", function () 

    local stash

    before_each( function ()

          setup()

          -- saotrace.config gets reset in setup(), so load the new one
          config = require('saotrace.config').raygen

          -- need to properly reload this for each test to get
          -- clean environment
          package.loaded['chandra.saotrace.raygen.focus'] = nil
          package.loaded['chandra.saotrace.raygen.source'] = nil

          stash = intrinsic.stash

    end)


    it( "focus_default", function ()

           config.scripts.entrance_aperture.args = { file = ea_db, shell = 1 }
           config.scripts.sources.args = { spectrum = 1.49 }
           raygen.init( 'chandra.saotrace.raygen.focus' )


           -- start the source
           assert.same( { { name = 'point1'},
                 func = 'start',
                 mod = 'source'
                        },
              stash[1])

           -- the order of the following is dependent upon what happens in
           -- saotrace.raygen.source.create

           assert.same( 'extent', stash[2].mod );
           assert.same( 'point',    stash[2].func );
           assert.same( 0,    stash[2][1].position.theta[1] );
           assert.same( 0,    stash[2][1].position.phi[1] );

           assert.same( 'spectrum', stash[3].mod );
           assert.same( 'mono',     stash[3].func );
           assert.same( 1.49,       stash[3][1].energy );
           assert.same( 1,          stash[3][1].flux );

           -- finish the source
           assert.same( { { name = 'point1' },
                 func = 'finish',
                 mod = 'source'
                        },
              stash[4])


           -- check the entrance aperture
           assert.same( 'ea',       stash[#stash].mod );
           assert.same( 'annulus',  stash[#stash].func );
           assert.same( 10,         stash[#stash][1].opts.ro );

    end)

    it( "focus_spectrum", function ()

           raygen.init( 'chandra.saotrace.raygen.focus',
                        string.format( argfmt, 'spectrum = 44')
           )

           -- start the source
           assert.same( { { name = 'point1'},
                 func = 'start',
                 mod = 'source'
                        },
              stash[1] )

           -- the order of the following is dependent upon what happens in
           -- saotrace.raygen.source.create

           assert.same( 'extent', stash[2].mod );
           assert.same( 'point',    stash[2].func );
           assert.same( 0,    stash[2][1].position.theta[1] );
           assert.same( 0,    stash[2][1].position.phi[1] );

           assert.same( 'spectrum', stash[3].mod );
           assert.same( 'mono',     stash[3].func );
           assert.same( 44,         stash[3][1].energy );
           assert.same( 1,          stash[3][1].flux );

           -- finish the source
           assert.same( { { name = 'point1' },
                 func = 'finish',
                 mod = 'source' },
              stash[4])

           -- check the entrance aperture
           assert.same( 'ea',       stash[#stash].mod );
           assert.same( 'annulus',  stash[#stash].func );
           assert.same( 10,         stash[#stash][1].opts.ro );


    end)

    it( "focus_position", function ()

           local theta, phi = 20, 3

           raygen.init( 'chandra.saotrace.raygen.focus',
                        string.format( argfmt, 'position = { theta = '
                                          .. theta .. ', phi = '
                                          .. phi .. ' }; spectrum = 1.49')
           )

           theta, phi = chandra_coords.MSC_to_raygen_polar( theta * min2rad , phi * deg2rad )

           -- start the source
           assert.same( { { name = 'point1'},
                 func = 'start',
                 mod = 'source'
                        },
              stash[1]
           )

           -- the order of the following is dependent upon what happens in
           -- saotrace.raygen.source.create

           assert.same( 'extent', stash[2].mod );
           assert.same( 'point',    stash[2].func );
           assert.same( theta, convert( stash[2][1].position.theta, nil, units.angular.radian)[1] );
           assert.same( phi,   convert( stash[2][1].position.phi, nil, units.angular.radian)[1] );

           assert.same( 'spectrum', stash[3].mod );
           assert.same( 'mono',     stash[3].func );
           assert.same( 1.49,       stash[3][1].energy );
           assert.same( 1,          stash[3][1].flux );

           -- finish the source
           assert.same( { { name = 'point1' },
                 func = 'finish',
                 mod = 'source' },
              stash[4])

           -- check the entrance aperture
           assert.same( 'ea',       stash[#stash].mod );
           assert.same( 'annulus',  stash[#stash].func );
           assert.same( 10,         stash[#stash][1].opts.ro );


    end)
end)
