setup = require( 'setup' )

local spec = { require( 'chandra.saotrace.raygen.sources.position' ).vspec }
local vobj = require( 'saotrace.raygen.validate' ).vobj:new()

vobj:setopts{ named = false }

local units   = require('saotrace.raygen.units')
local convert = units.convert

local chandra_coords   = require( 'chandra.saotrace.coords' )
local raygen_coords   = require( 'saotrace.raygen.coords' )
local MSC_to_raygen_polar = chandra_coords.MSC_to_raygen_polar
local raygen_elaz_2_raygen_polar = raygen_coords.raygen_elaz_2_raygen_polar

local min2rad = raygen_coords.min2rad
local deg2rad = raygen_coords.deg2rad

describe( "position", function ()

             before_each( setup )

    it( "position_default", function ()

       local theta, phi = 3, 2

       local ok, val = vobj:validate( spec,
                                 {
                                    theta = theta,
                                    phi = phi,
                                 })
       assert.is_true( ok )

       theta, phi = MSC_to_raygen_polar( theta * min2rad, phi * deg2rad )

       assert.same( theta, convert( val.theta, nil, units.angular.radian)[1] )
       assert.same( phi,   convert( val.phi,   nil, units.angular.radian)[1] )


    end )

    it( "position_msc", function ()

       local theta, phi = 3, 2

       local ok, val = vobj:validate( spec,
                                 {
                                    theta = theta,
                                    phi = phi,
                                    coord = 'chandra-msc'
                                 })

       theta, phi = MSC_to_raygen_polar( theta * min2rad, phi * deg2rad )

       assert.is_true( ok )

       assert.same( theta, convert( val.theta, nil, units.angular.radian)[1] )
       assert.same( phi,   convert( val.phi,   nil, units.angular.radian)[1] )

    end)


    it( "position_offset", function ()

       local zoff = 3
       local yoff = 2

       local theta, phi = raygen_elaz_2_raygen_polar( -zoff * min2rad, yoff * min2rad )

       local ok, val = vobj:validate( spec,
                                 {
                                    zoff = zoff,
                                    yoff = yoff,
                                    coord = 'chandra-offset'
                                 })

       assert.is_true( ok )

       assert.same( theta, convert( val.theta, nil, units.angular.radian)[1] )
       assert.same( phi,   convert( val.phi,   nil, units.angular.radian)[1] )


       -- now check if coord is not specified
       local ok, val = vobj:validate( spec,
                                 {
                                    zoff = zoff,
                                    yoff = yoff,
                                 })

       assert.is_true( ok )


    end)


    it( "equatorial", function ()

       local position = { ra = '22:08:40.828',
                          dec = '+45:44:32.55',
                          ra_aimpt = 332.16395709286,
                          dec_aimpt = 45.740257117459,
                          coord = 'chandra-equatorial'
                       }

       local theta = { 17.3, 'arcsecond' }
       local phi = { 333.77, 'degree' }

       local ok, val = vobj:validate( spec, position )

       assert.is_true( ok )

       assert.near( theta[1], convert( val.theta, nil, units.angular.arcsecond)[1], .1 )
       assert.near( phi[1],   convert( val.phi,   nil, units.angular.degree)[1], .01 )

       -- now check if coord is not specified

       position.coord = nil
       local ok, val = vobj:validate( spec, position )
       assert.is_true( ok )

    end)

end)
