!+
!KWIC bessel.f
!
!$Id: bessel.f,v 1.1 2004/03/16 15:49:44 dtn Exp $
!
!Revisions:
!   93-Oct-15[T. Gaetz]
!      . eliminate ascii io to unit 6; unit 6 now feeds binary pipe and
!        should be clear of ascii output; see unit 8 for messages
!-

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     SUBROUTINE BESSJN(X,N,BJN,ERRA, IER)
C
C
C
C    PURPOSE
C        COMPUTE THE ORDINARY
C        BESSEL FUNCTIONS OF ORDER N
C              J (X)
C               N
C
C    USAGE
C         CALL BESSJN (X, N, BJN, ERRA, IER)
C
C    PARAMETERS
C         BJN  - VALUE OF BESSEL FUNCTION
C         ERRA - ACCURACY
C         IER  - ERROR FLAG
C              = 1 => FATAL ERROR BESSJN COULD NOT ACHIEVE DESIRED
C                     PRECISION
C
C         N    - ORDER
C         X    - ARGUMENT
C
C    METHOD
C         USING THE RATIO X/DFLOAT(N)
C         BESSJN DIRECTS THE CALCULATION TO ONE OF 3 SUBROUTINES
C     X/N < 0.50  => CALL BESSJN0 FOR SUMMATION OF ASCENDING SERIES
C     X/N > 100. => CALL BESSJN1 FOR SUMMATION OF ASYMPTOTIC SERIES
C      OTHERWISE  => CALL BESSJN2 FOR DOWNWARDS RECURSION
C
C
C
C
C    WRITTEN 4/12/84 BY JOHN PARKER OF SSAI UNDER CONTRACT WITH
C    GSFC.
C
C    REFERENCE: ABRAMOWITZ AND STEGUN, HANDBOOK OF MATHEMATICAL
C                FUNCTIONS (NATIONAL BUREAU OF STANDARDS)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      SUBROUTINE BESSJN(X,N,BJN,ERRA, IER)
      IMPLICIT DOUBLE PRECISION (A-H, O-Z)
      IMPLICIT INTEGER (I-N) 
      DATA ERRMIN/1.D-13/, ZERO/0.0D0/, ONE/1.0D0/
      DATA TEN/10.0D0/
C
      IER = 0
      IER2 = 0
      IER1 = 0
      BJN = ZERO
      ERR = DABS(ERRA)
      IF(ERR .LE. ERRMIN) ERR= ERRMIN
C
C        DETERMINE PHASE
C
        PHASE = ONE
      IF(((X .LT. ZERO) .OR. (N .LT. 0)) .AND.
     A   (MOD(N,2) .NE. 0)) PHASE = -ONE
      IF ((X .LT. ZERO) .AND. (N.LT.0)) PHASE = ONE
C
C        CHANGE INPUT VBL NAME SO INPUT NOT DESTROYED
C
      XA  = DABS(X)
      NA  = IABS(N)
      DNA1= DFLOAT(NA+1)
C
C        CHECK FOR SPECIAL VALUES
C
      IF( XA .GT. ZERO) GO TO 10
C
C         SPECIAL VALUES FOR X = 0
C
      BJN = ZERO
      IF( NA .EQ. 0) BJN = ONE
      GO TO 50
C
   10  CONTINUE
       TEST = XA/DNA1
       IF(TEST   .LE. 0.50D0) GO TO 20
       IF(TEST   .GE. 1.D02 ) GO TO 30
   12  CALL BESSJN2 (XA, NA, BJN, ERR, IER)
       IF(IER .EQ. 0) GO TO 50
       IF( (IER2.NE.0) .OR. (IER3.NE.0)) GO TO 40
C
C    DOWNWARDS RECURSION FAILED TO CONVERGE.  TRY ASCENDING OR
C    ASYMPTOTIC SERIES
C
       IF(TEST.LE.ONE) GO TO 20
       IF(TEST.GT.TEN) GO TO 30
       GO TO 40
   20  CALL BESSJN0(XA, NA, BJN, ERR, IER2)
       IF(IER2.EQ.0) GO TO 50
       IF(IER.EQ.0) GO TO 12
       IER = IER2
       GO TO 40
   30  CALL BESSJN1(XA, NA, BJN, ERR, IER3)
       IF(IER3.EQ.0) GO TO 50
       IF(IER.EQ.0) GO TO 12
       IER = IER3
       GO TO 40
   40  CONTINUE
       WRITE(8,10010) IER, X, N
10010  FORMAT(/,' BESSJN: ERROR. FAILURE TO CONVERGE',/
     A          ' BESSJN:                 ERROR FLAG IER = ',I5,
     B        /,' BESSJN:',20X,'(X, N) =    ',1P,D15.6,I5,/)
   50  RETURN
       END
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      SUBROUTINE BESSJN0 (X, N, BJN, ERR, IER)
      IMPLICIT DOUBLE PRECISION (A-H, O-Z)
      IMPLICIT INTEGER (I-N) 
      DATA ZERO/0.0D0/, ONE/1.0D0/
C
C       USE ASCENDING SERIES FOR SMALL ARGUMENTS
C
      IF(IER .EQ. 100) WRITE(8, 1)
 1    FORMAT(10X,' BESSEL FUNCTION ROUTINE: ASCENDING SERIES',
     A //, ' NUM. ', '   TERM VALUE  ', '          SUM  '/)
           M= 0
           XHALF = X/2.0D0
           Y     = -XHALF*XHALF
           DN    = DFLOAT(N)
           DN1   = DFLOAT(N+1)
           TERM  = ONE
           BJN   = TERM
   20 IF(M.GT.500) GO TO 50
              M=M+1
              DM = DFLOAT(M)
              TERM  = TERM * Y /(DM*(DM + DN))
              BJN = BJN + TERM
              IF(DABS(TERM/BJN).GT.ERR) GO TO 20
      IF(N.EQ.0) GO TO 40
      ATMP = ONE
      DO 30 I=1,N
          DI = DFLOAT(I)
          ATMP = ATMP * (XHALF/DI)
   30 CONTINUE
      BJN = BJN * ATMP
   40 RETURN
   50 IER = M
      RETURN
      END
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      SUBROUTINE BESSJN1 (X, N, BJN, ERR, IER)
      IMPLICIT DOUBLE PRECISION (A-H, O-Z)
      IMPLICIT INTEGER (I-N) 
      DATA ZERO/0.0D0/, ONE/1.0D0/
      DATA PI/3.141592653589793238462643D0/
C
C     ASYMPTOTIC SERIES FOR BJN
C
      PHASE = ONE
       DN   = DFLOAT(N)
       DMU = DN*DN *4.0D0
       EIGHTX = 8.0D0*X
       KTEST =  (N+N+1)/4
       TERM = ONE
       P= ONE
       Q = ZERO
       K = 0
  230  K=K+1
          OLDTRM = TERM
          DK = DFLOAT(K)
          TERM = TERM*(DMU-(DK+ONE)*(DK+ONE))/(DK*EIGHTX)
          IF(MOD(K,2) .NE.0) GO TO 120
            PHASE = -PHASE
            P = P + PHASE * TERM
          GO TO 130
C
C
  120     Q = Q + PHASE*TERM
          GO TO 130
  130 CONTINUE
C     WRITE(8,131) K,TERM,P,Q   ! change to unit 8:
C 131 FORMAT('  K,TERM,P,Q=',I5,1P,3D20.10,/)
      IF(K.LT.KTEST) GO TO 230
      IF(P.NE.ZERO) ATMP = DABS(TERM/P)
      IF(P.EQ.ZERO) ATMP = DABS(TERM)
      IF ((ATMP .GT. ERR)
     A       .AND.(DABS(TERM) .LT. DABS(OLDTRM)))GO TO 230
C
C
C
      IF(DABS(TERM).GT.DABS(OLDTRM)) GO TO 140
C
C    ASYMPTOTIC SERIES SUCCESSFULLY CONVERGED
C
       CHI = X - (DN +DN+ONE)*PI*0.25D0
       BJN = P*DCOS(CHI)-Q*DSIN(CHI)
       BJN = BJN*DSQRT(2.0D0/(X *PI))
       RETURN
C
C   ASYMPTOTIC SERIES FAILED TO CONVERGE
C
 140   IF(K.LT.KTEST) GO TO 230
       IER = 1
       RETURN
       END
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     SUBROUTINE BESSJN2(X,N,BJN,ERRA, IER)
C
C
C
C    PURPOSE
C        COMPUTE THE ORDINARY
C        BESSEL FUNCTIONS OF ORDER N
C              J (X)      BY DOWNWARDS RECURSION
C               N
C
C    USAGE
C         CALL BESSJN (X, N, BJN, ERRA, IER)
C
C    PARAMETERS
C         BJN  - VALUE OF BESSEL FUNCTION
C         ERRA - ACCURACY
C         IER  - ERROR FLAG
C              = 1 => FATAL ERROR BESSJN COULD NOT ACHIEVE DESIRED
C                     PRECISION
C              =100=> DETAILED PRINTS
C         N    - ORDER
C         X    - ARGUMENT
C
C    METHOD
C        STARTING AT A LARGE ORDER, RECUR DOWNWARDS
C              USING
C      K (X) = 2*(N+1)*K   (X)/X - K(X)
C       NN+1N+2
C
C      THE K'S ARE EQUAL TO J'S WITHIN A CONSTANT FACTOR DETERMINED BY
C       DNORM = K  + 2*(K  + K  + . . . )
C                0       2    4
C
C      THEN  J  = K  /DNORM
C             N    N
C
C    WRITTEN 4/12/84 BY JOHN PARKER OF SSAI UNDER CONTRACT WITH
C    GSFC.
C
C    REFERENCE: ABRAMOWITZ AND STEGUN, HANDBOOK OF MATHEMATICAL
C                FUNCTIONS (NATIONAL BUREAU OF STANDARDS)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      SUBROUTINE BESSJN2 (X, N, BJN, ERR, IER)
      IMPLICIT DOUBLE PRECISION (A-H, O-Z)
      IMPLICIT INTEGER (I-N) 
      DATA ZERO/0.0D0/, ONE/1.0D0/
C
      BJN    = ZERO
      BSAVE  = ZERO
      NSTART = MAX0(N + 10, IDINT(X + 10.0D0) )
      ICHECK = 0
C
C FOR A GIVEN NSTART, THIS LOOP IS PASSED THROUGH TWICE
C    TO CHECK ERRORS
C
   40 M = NSTART
      DNORM  = ZERO
         BJMP1 = ZERO
         BJMP2 = ZERO
         BJM   = 1.D-28
C               RECURSION
   50    IF(M .LE. 0) GO TO 60
              M=M-1
              DM = DFLOAT(M)
C  C
              BJMP2 = BJMP1
              BJMP1 = BJM
C  C
              BJM   =(2.0D0 * (DM+ONE ) * BJMP1/X)- BJMP2
C
              IF (M.EQ. N) BJN = BJM
              IF (MOD(M,2).NE.0) GO TO 50
              IF (M .NE. 0) DNORM = BJM + DNORM + BJM
              IF (M .EQ. 0) DNORM = BJM + DNORM
          GO TO 50
C                NORMALIZE
   60         BJN = BJN/DNORM
              IF( ICHECK.EQ.1) GO TO 70
                 ICHECK = 1
                 BSAVE = BJN
                 NSTART = NSTART + 2
                 GO TO 40
C
C          ACCURATE???
C
   70        IF(DABS(BSAVE-BJN) .LE. DABS(BJN*ERR)) GO TO 90
C
C           TRY AGAIN
C
   80            BSAVE = BJN
                 ICHECK = 1
                 NSTART = NSTART  + 5
                 IF(NSTART-N  .LT. 100 ) GO TO 40
C
C       FATAL  ERROR TRAP
C
                     IER = NSTART
   90 RETURN
      END
