/****************************************************************************
 *
 * Source     : osacbpipe.c
 * Author     : Terry Gaetz
 *
 * Description: handle bpipe stuff
 *
 * History
 *--------
 * 0.0.0  1996-Feb-12 tjg  original version
 *
 * $Id: osacbpipe.c,v 1.4 2004/07/16 18:57:46 dtn Exp $
 */

#include <unistd.h> 
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include <tracefct/tracefct.h>
#include <tracefct/exiterrvals.h>
#include <mathconst/mathconst.h>    

#include <bpipe/datatypes.h>
#include <bpipe/bpipe.h>
#include <bpipe/bpipe_axaf.h>

#include "config.h"

#include "osacbpiperay.h"
#include "osacbpipe.h"

#include "field_struct.h"   

/*
 * static methods
 */
typedef struct
{
  BPipe       *bpipe;         /* the BPipe */
  BPipeOutput *outpipe;       /* output bpipe */

  char        *core_image;    /* core image for BPipe */
  size_t       s_core_image;  /* sizeof(*core_image)  */

} osacBPipe;

static osacBPipe *new_osacBPipe(char *in, char *out, Ray *srcRay, Ray *dstRay, 
				int *db_extend);
static void       delete_osacBPipe(osacBPipe *This );
static size_t     next_ray(osacBPipe *This);
static DpktField *check_field_osacBPipe( osacBPipe *This, char *name );
static void      *get_dataptr_osacBPipe( osacBPipe *This, char *name,
					 void *in, void *out, int n_fields, Fieldinfo *field_info );
static void       create_fields_osacBPipe( osacBPipe *This,
		     Fieldinfo *field_info, int n_fields, int *db_extend);
static void       attach_ray_osacBPipe( osacBPipe *This, Ray *srcRay, 
		       Ray *dstRay, int n_fields, Fieldinfo *field_info  );
static DpktField  *extend_field_osacBPipe( osacBPipe *This, char *name );
static Fieldinfo  *find_field_info( char *name, Fieldinfo *field_info,
				    int n_fields );

/*
 * static data
 */

#define NAME_LENGTH 255

static int        theSurfNo;
static char       inpipe_name[NAME_LENGTH];
static char       outpipe_name[NAME_LENGTH];
static osacBPipe *theBPipe;

static Ray        srcRay;    /* input  ray */
static Ray        dstRay;    /* output ray */

static int 	  goodrays_flag;  

/*===========================================================================
 * Fortran interface
 */


/*---------------------------------------------------------------------------
 * attach_bpipe_ - acquire & attach the BPipe; check it out (fortran interface)
 */
void
ATTACH_BPIPE_F77(char *in, char *out, int *db_extend,
		 int *fgoodrays, mst_fortran_charlen_t ln_in, mst_fortran_charlen_t ln_out)
{
  strncpy(inpipe_name, in, (size_t)ln_in);
  inpipe_name[ln_in] = '\0'; 
  strncpy(outpipe_name, out, (size_t)ln_out);
  outpipe_name[ln_out] = '\0'; 
  
  goodrays_flag = *fgoodrays ;
  theBPipe = new_osacBPipe(inpipe_name, outpipe_name, &srcRay, &dstRay,
			   db_extend);
}

/*---------------------------------------------------------------------------
 * detach_bpipe_ - detach the BPipe; free resources
 */
void
DETACH_BPIPE_F77(void)
{
  delete_osacBPipe(theBPipe);
}

/*---------------------------------------------------------------------------
 * get_ray_ - read next ray from BPipe...
 */
void
GET_RAY_F77(double   *r,            /* ray position  vector (3-vector)      */
	    double   *v,            /* ray direction vector (unit 3-vector) */
	    double   *wt,           /* ray weight                           */
	    double   *energy,       /* ray energy                           */
	    DComplex *c2,           /* polarization amplitude ("cosine")    */
	    DComplex *s2,           /* polarization amplitude ("sine")      */
	    int      *id,           /* ray identifier             srcRay    */
	    int      *raycode,      /* ray success code                     */
	    int      *err           /* 0 for success, nonzero for error     */
	    )
{
  size_t nrays;
  
  /*
   * obtain data from bpipe...
   */
  nrays = next_ray(theBPipe);
  
  *err = (nrays == 1) ? 0 : 1;
  
  /*
   * transfer the data from the ray struct to the argument fields...
   */
  *id       = *srcRay.rayid;
  *wt       = *srcRay.wt;
  *energy   = *srcRay.energy;
  *raycode  = *srcRay.raycode;
  
  r[0] = srcRay.pos->x;
  r[1] = srcRay.pos->y;
  r[2] = srcRay.pos->z;
  v[0] = srcRay.dir->x;
  v[1] = srcRay.dir->y;
  v[2] = srcRay.dir->z;
  
  c2[0].r = srcRay.c2->q1.r;         /* 'cosine' component */
  c2[0].i = srcRay.c2->q1.i;         /* 'cosine' component */
  c2[1].r = srcRay.c2->q2.r;         /* 'cosine' component */
  c2[1].i = srcRay.c2->q2.i;         /* 'cosine' component */
  s2[0].r = srcRay.s2->q1.r;         /* 'sine'   component */
  s2[0].i = srcRay.s2->q1.i;         /* 'sine'   component */
  s2[1].r = srcRay.s2->q2.r;         /* 'sine'   component */
  s2[1].i = srcRay.s2->q2.i;         /* 'sine'   component */
  
}

/*---------------------------------------------------------------------------
 * put_ray_ - push ray to BPipe...
 */
void
PUT_RAY_F77(double   *r,            /* ray position  vector (3-vector)      */
	    double   *v,            /* ray direction vector (unit 3-vector) */
	    double   *norm,         /* surface normal at intercept          */
	    double   *intersect_bcs,/* intersection at BCS optics           */
	    double   *wt,           /* ray weight                           */
	    double   *energy,       /* ray energy                           */
	    double   *g_ang,        /* graze angle                          */
	    DComplex *c2,           /* polarization amplitude ("cosine")    */
	    DComplex *s2,           /* polarization amplitude ("sine")      */
	    int      *rayid,        /* ray identifier                       */
	    int      *raycode,      /* ray success code                     */
	    int      *ray_missed    /* surface hit code:  0 = hit, 1 = miss */
	    )
{
   if ( *ray_missed )
     {
       if ( goodrays_flag )
	 {
	   return;    /* short circuit */
	 }
       /* set appropriate bit in dstRay.missed since it's a ghost ray */
       
       BITSET((*dstRay.missed), theSurfNo);
       
      /* 
       * transfer the data to the ray struct. It's a ghost ray so copy old values
       */
       *dstRay.wt      = *srcRay.wt;
       *dstRay.energy  = *srcRay.energy;
       *dstRay.g_ang   = *srcRay.g_ang;       
       *dstRay.pos     = *srcRay.pos;
       *dstRay.dir     = *srcRay.dir;
       *dstRay.norm    = *srcRay.norm;
       *dstRay.intersect_bcs    = *srcRay.intersect_bcs;
       dstRay.c2->q1.r = srcRay.c2->q1.r;
       dstRay.c2->q1.i = srcRay.c2->q1.i;
       dstRay.c2->q2.r = srcRay.c2->q2.r;
       dstRay.c2->q2.i = srcRay.c2->q2.i;
       dstRay.s2->q1.r = srcRay.s2->q1.r;
       dstRay.s2->q1.i = srcRay.s2->q1.i;
       dstRay.s2->q2.r = srcRay.s2->q2.r;
       dstRay.s2->q2.i = srcRay.s2->q2.i;   
     }
  else
    {
      /* 
       * transfer the data to the ray struct (not a ghost so update)
       */
      *dstRay.wt       = *wt;
      *dstRay.energy   = *energy;
      *dstRay.g_ang    = *g_ang;       

      dstRay.pos->x  = r[0];
      dstRay.pos->y  = r[1];
      dstRay.pos->z  = r[2];
      dstRay.dir->x  = v[0];
      dstRay.dir->y  = v[1];
      dstRay.dir->z  = v[2];
      dstRay.norm->x = norm[0];
      dstRay.norm->y = norm[1];
      dstRay.norm->z = norm[2];
      dstRay.intersect_bcs->x = intersect_bcs[0];
      dstRay.intersect_bcs->y = intersect_bcs[1];
      dstRay.intersect_bcs->z = intersect_bcs[2];
      
      dstRay.c2->q1.r = c2[0].r;         /* 'cosine' component */
      dstRay.c2->q1.i = c2[0].i;         /* 'cosine' component */
      dstRay.c2->q2.r = c2[1].r;         /* 'cosine' component */
      dstRay.c2->q2.i = c2[1].i;         /* 'cosine' component */
      dstRay.s2->q1.r = s2[0].r;         /* 'sine'   component */
      dstRay.s2->q1.i = s2[0].i;         /* 'sine'   component */
      dstRay.s2->q2.r = s2[1].r;         /* 'sine'   component */
      dstRay.s2->q2.i = s2[1].i;         /* 'sine'   component */
      
    }
  /*
   * push the ray into the output pipe
   */

  *dstRay.raycode  = *raycode;  /* here because true for all rays */
  if ( bpipe_write_dpkt(theBPipe->bpipe, theBPipe->core_image, 
                        theBPipe->outpipe) ) {
    tf_enter("put_ray_");  
    tf_exit(ExitERR_fread, "error in put_ray_ while writing ray %d to `%s'",
	    *rayid, outpipe_name,"bpipe errno is %s",
	    bpipe_strerror(bpipe_errno));
  }
  *dstRay.missed = 0;  /* initialize to neutral state for later rays */
}

/*===========================================================================
 * static methods
 */

/*---------------------------------------------------------------------------
 * new_osacBPipe -
 */
static osacBPipe*
new_osacBPipe(char *in, char *out, Ray *srcRay, Ray *dstRay, int *db_extend)
{
 
  osacBPipe     *This;
  int n_fields;
  int *srfptr;  
  unsigned int surface_num;
  
  /*
    set up field structure:
    field name, BPipe data type, extendable?(1 = yes), extended?(1 = yes)
    */

  Fieldinfo field_info[] = { 
    { BP_AXAF_phot_pos,         BPDType_DVector3,  1, 0 },
    { BP_AXAF_phot_dir,         BPDType_DVector3,  1, 0 },
    { BP_AXAF_surf_norm,        BPDType_DVector3,  1, 0 },
    { BP_AXAF_intersect_bcs,    BPDType_DVector3,  1, 0 },
    { BP_AXAF_phot_energy,      BPDType_double,    0, 0 },
    { BP_AXAF_phot_graze_angle, BPDType_double,    1, 0 },
    { BP_AXAF_phot_id,          BPDType_uint,      0, 0 },
    { BP_AXAF_phot_wt,          BPDType_double,    1, 0 },
    { BP_AXAF_pol_cos,          BPDType_DCVector2, 1, 0 },
    { BP_AXAF_pol_sin,          BPDType_DCVector2, 1, 0 },
    { BP_AXAF_osac_raycode,     BPDType_int,       0, 0 },
    { BP_AXAF_surf_missed,      BPDType_uint,      0, 0 }
  };
  /* get number of fields */
  n_fields = (sizeof(field_info)/sizeof(Fieldinfo));
  tf_enter("new_osacBPipe"); 
  
  if ( 0 == (This = (osacBPipe*)malloc(sizeof(osacBPipe))   ))
    {
      tf_exit(ExitERR_misc,  "malloc failure; can't malloc This, bpipe errno is %s:", bpipe_strerror(bpipe_errno));
    }
  /*
   * open bpipes...
   */
  if ( (This->bpipe = bpipe_new())==NULL ) 
    {
      tf_exit(ExitERR_misc,"bpipe_new failure; bpipe errno is %s",
	      bpipe_strerror(bpipe_errno));
    }
  
  /*
   * ... set up input bpipe...
   */
  if ( bpipe_input(This->bpipe, in) ) 
    {
      tf_exit(ExitERR_misc,  "bpipe_input failure; bpipe errno is %s", 
	      bpipe_strerror(bpipe_errno));
    }
  
  /*
   * ... set up output bpipe...
   */
  if ( (This->outpipe=bpipe_output(This->bpipe, out)) == NULL )
    {
      tf_exit(ExitERR_misc, "bpipe_output failure; bpipe errno is %s",
	      bpipe_strerror(bpipe_errno));
    }
  
  /*
   * create fields in data packet 
   */
  create_fields_osacBPipe( This, field_info, n_fields, db_extend);
  
  /*
   * add surface number field if not present and initialize
   */
  
  if ( 0 == bpipe_hdrf_n( This->bpipe, "srfno"))
    {
      surface_num = 0;    
      if ( bpipe_hdrf_add( This->bpipe,"srfno", BPDType_uint, NULL,
			   &surface_num, 1))
	{
	  tf_exit(ExitERR_misc, "bpipe_hdrf_add failure; bpipe errno is %s:",
		  bpipe_strerror(bpipe_errno));  
	}
    }
  srfptr = (int *) (bpipe_hdrf_data (This->bpipe, "srfno", (size_t)0));
  (*srfptr)++;
  theSurfNo = (*srfptr) - 1;
  
  /*
   * set up map...
   */
  if ( 0 == (This->s_core_image=bpipe_map(This->bpipe)) ) 
    {
      tf_exit(ExitERR_misc, "bpipe_map failure; bpipe errno is %s", 
	      bpipe_strerror( bpipe_errno)); 
    }
  
  if ( 0 == (This->core_image = malloc(This->s_core_image)) )
    {
      tf_exit(ExitERR_misc,  "can't malloc core_image, bpipe errno is %s", 
	      bpipe_strerror( bpipe_errno));
    }
  
  /*
   * write header information to output..
   */
  if ( 0 != (bpipe_write_hdr(This->bpipe)) )
    {
      tf_exit(ExitERR_misc,  "bpipe_write_hdr failure; bpipe errno is %s: ",
	      bpipe_strerror(bpipe_errno));
    }
  
  /*
   * attach data fields to ray structure...
   */
  attach_ray_osacBPipe( This, srcRay, dstRay, n_fields, field_info );
  
  tf_leave();
  
  return This;
}

/*---------------------------------------------------------------------------
 * delete_osacBPipe - free up the resources 
 */
static void delete_osacBPipe(osacBPipe *This)
{
  bpipe_delete(This->bpipe);
  free(This->core_image);
  free(This);
  
}

/*---------------------------------------------------------------------------
 * next_ray - read next ray from BPipe...
 */
static size_t
next_ray(osacBPipe *This)
{
  return bpipe_read_dpkts(This->bpipe, This->core_image, (size_t)1);
}

/*---------------------------------------------------------------------------
 * get the data pointer for a named data packet field
 * and copy data from core image
 */
static void *
get_dataptr_osacBPipe( osacBPipe *This, char *name, void *src, void *dst, 
		       int n_fields, Fieldinfo *field_info )
{

  Fieldinfo *fi = find_field_info( name, field_info, n_fields );
  DpktField *field = check_field_osacBPipe( This, name );

  tf_enter("get_dataptr_osacBPipe");  

  if ( NULL == fi){
    tf_exit(ExitERR_misc,"error for find_field_info, bpipe error is %s",
	    bpipe_strerror(bpipe_errno)
	    );
  }
  *((void **) dst) = bpipe_dpktf_data( field, This->core_image );
  if ( fi->didextend ) 
    { 
      *((void **) src) = ((char *)*((void**) dst )) + 
	bpipe_datatype_size( fi->type) ; 
    }
  else
    {     
      *((void **) src) = *((void**) dst ) ;
    }
  
  /* initialize field */
  bpipe_dpktf_init( field, This->core_image, NULL );
  tf_leave();  
  
  return NULL;
}

/*---------------------------------------------------------------------------
 * simple routine to get the data pointer for a named data packet field
 */
static DpktField*
check_field_osacBPipe( osacBPipe *This, char *name )
{
  DpktField *field = bpipe_dpktf( This->bpipe, name );
  
  if ( field == NULL )
    {  
      tf_enter("check_field_osacBPipe"); 
      tf_exit( ExitERR_misc, "field `%s' doesn't exist in input data stream",
	       name, "bpipe error is  %s",  bpipe_strerror(bpipe_errno)
	       );
    }
  return field;
}

/*---------------------------------------------------------------------------
 * create each BPipe field and extend if appropriate
 */
static void
create_fields_osacBPipe( osacBPipe *This, Fieldinfo *field_info, int n_fields,
			 int *db_extend )
{
  Fieldinfo *fi = field_info;
  for (; fi < field_info + n_fields ; fi++)  
    { 
      int rc = bpipe_dpktf_add(This->bpipe, fi->name, fi->type, NULL );

      switch( rc )
	{
	case -1:  /* error creating field */	
	          tf_enter("create_fields_osacBPipe"); 
	          tf_exit( ExitERR_misc, 
                       "error creating data packet field: bpipe errno is  %s", 
                        bpipe_strerror( bpipe_errno) ); 
	          break;

	case 0:   /* successful creation; don't extend */
	          break;

	case 1:   /* field already exists; extend it if appropriate*/
	          if ( *db_extend ==1 ){ 
	            if ( fi->extend ){
	              extend_field_osacBPipe( This, fi->name );
	              fi->didextend = 1;
	            }
	          }
	          break;

	default: /* impossible to get here */

		  tf_message( "create_fields_osacBPipe: "
                              "bpipe_dpktf_add returned invalid value: %d,",
                              rc, " aborting." );
		  exit(1);
	  	  break;
	}
    }
}

/*---------------------------------------------------------------------------
 * attach BPipe fields to ray struct
 */
static void
attach_ray_osacBPipe( osacBPipe *This, Ray *srcRay, Ray *dstRay, 
		      int n_fields, Fieldinfo *field_info  )
{
  get_dataptr_osacBPipe(This, BP_AXAF_phot_pos, &srcRay->pos, 
			&dstRay->pos, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_phot_dir, &srcRay->dir,
			&dstRay->dir, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_surf_norm, &srcRay->norm, 
			&dstRay->norm, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_intersect_bcs, &srcRay->intersect_bcs, 
			&dstRay->intersect_bcs, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_phot_energy, &srcRay->energy, 
			&dstRay->energy, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_phot_graze_angle, &srcRay->g_ang,
			&dstRay->g_ang, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_phot_wt, &srcRay->wt,
			&dstRay->wt, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_pol_cos, &srcRay->c2,
			&dstRay->c2, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_pol_sin, &srcRay->s2,
			&dstRay->s2, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_phot_id, &srcRay->rayid, 
			&dstRay->rayid, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_osac_raycode, &srcRay->raycode, 
			&dstRay->raycode, n_fields, field_info );
  get_dataptr_osacBPipe(This, BP_AXAF_surf_missed, &srcRay->missed, 
			&dstRay->missed, n_fields, field_info );
}

/*---------------------------------------------------------------------------
 * extend a given field
 */
static DpktField*
extend_field_osacBPipe( osacBPipe *This, char *name )
{
  size_t *dst_off;
  BPMatrix *bpmatrix;
  DpktField *field = check_field_osacBPipe( This, name );

  tf_enter("extend_field_osacBPipe");    
  
  /* get matrix description for core image */
  if ( NULL == ( bpmatrix = bpipe_dpktf_matrix ( field, BPDSite_CORE, NULL )))
    {            
      tf_exit(ExitERR_misc, "error getting matrix spec; bpipe errno is %s",
	      bpipe_strerror(bpipe_errno));
    }
  
  /* ... and now extend it */
  bpmatrix->extent[0]++;     
  
  /* get offset for destination array */
  if ( NULL == (  dst_off = bpipe_offset_new_va( (size_t)1, 1) ) ) 
    {
      tf_exit(ExitERR_misc, "  error from bpipe_offset_new_va is %s",
	      bpipe_strerror(bpipe_errno));
    }
  
  /* resize core image */
  if (bpipe_dpktf_resize_core( field, bpmatrix, NULL, dst_off, NULL))
    {
      tf_exit(ExitERR_misc, "error resizing core: bpipe errno is %s",
	      bpipe_strerror(bpipe_errno));
    }
  tf_leave();
  return field;
}

/*---------------------------------------------------------------------------
 *  find field info for given field 
 */
static Fieldinfo*
find_field_info( char *name, Fieldinfo *field_info, int n_fields  )
{
  Fieldinfo *fi =  field_info;
  for ( ; fi < field_info + n_fields ; fi++)  
    {
      if ( 0 == strcmp( name, fi->name ))
	return fi;
    }
  return NULL;
}



