/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of src
 *
 * src is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * src is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <config.h>
#include <libgen.h>

#include <cxcparam/parameter.h>
#include <tracefct/tracefct.h>
#include <tracefct/exiterrvals.h>
#include <mathconst/mathconst.h>
#include <suplib/debug.h>

#include "getpar.h"

#include "fproto.h"


static void warning_oldversion( void ) {

  tf_enter( "warning_oldversion" );
  tf_message( "surface_intercept: WARNING!\nsurface_intercept: You are "
	      "using an old version of the parameter file.\n" );
  tf_leave( );

}

static void willbeassumed( char msg[] ) {

  tf_enter( "willbeassumed" );
  tf_message( "surface_intercept: %s will be assumed.\n", msg );
  tf_leave( );

}

static void backcompatible_updateparfile( void ) {

  tf_enter( "backcompatible_updateparfile" );
  tf_message( "surface_intercept: Results will be backwards compatible.\n"
	      "surface_intercept: Please update the parameter file!" );
  tf_leave( );

}

paramfile open_paramfile( char* fname, int argc, char* argv[], char* mode ) {

  paramfile pfile      = NULL;

  pfile = paramopen( fname, argv, argc, mode );

  if ( NULL == pfile ) {

    if ( NULL == fname )
      tf_exit( ExitERR_fopen, "Unable to open file %s.par", basename( argv[0] ) );
    else
      tf_exit( ExitERR_fopen, "Unable to open file %s", fname );

  }

  return pfile;

}

void get_parameters( paramfile pfile, int srfno, int* db_extend,
		     char* input, char* output ) {

  char msg[ SZ_PFLINE ];

  char* fnm_gi         = NULL;
  char* fnm_dfm        = NULL;
  char* log_file       = NULL;
  char* fnm2_dfm       = NULL;
  char* debug          = NULL;
  int found_dfm_type   = 0;
  int dfm_type         = -1;
  int do_osac_reflect  = -1;
  double deg2rad       = M_DEG2RAD;
  double theta0        = 0.0;
  double theta02       = 0.0;
  double dfm_scale     = 0.0;
  double dfm2_scale    = 0.0;

  tf_enter( "get_parameters" );

  /* The user just want to get the version number, print then exit
  if ( pgetb( pfile, "version" ) ) {
    fprintf( stdout, "suface_intercept\t%s\n", VERSION );
    paramclose( pfile );
    tf_leave( );
    free( input );
    free( output );
    exit( 0 );
  }
  */

  /*
C
C   READ IN gi & FIRST DEFORMATION FILENAMES
C
   */
  fnm_gi = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == fnm_gi ) {
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'fnm_gi'" );
  }
  if ( NULL == pgetstr( pfile, "gi_filename", fnm_gi, SZ_PFLINE ) ) {
    free( fnm_gi );
    tf_exit( ExitERR_IOerr, "Unable to get 'gi_filename' parameter" );
  }

  fnm_dfm = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == fnm_dfm ) {
    free( fnm_gi );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'fnm_dfm'" );
  }
  if ( NULL == pgetstr( pfile, "dfm_filename", fnm_dfm, SZ_PFLINE ) ) {
    free( fnm_gi );
    free( fnm_dfm );
    tf_exit( ExitERR_IOerr, "Unable to get 'dfm_filename' parameter" );
  }

  /*
C
C   READ IN NAME OF logfile
C
*/
  log_file = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == log_file ) {
    free( fnm_gi );
    free( fnm_dfm );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'log_file'" );
  }
  if ( paccess( pfile, "logfile" ) ) {

    if ( NULL == pgetstr( pfile, "logfile", log_file, SZ_PFLINE ) ) {
      free( fnm_gi );
      free( fnm_dfm );
      free( log_file );
      tf_exit( ExitERR_IOerr, "Unable to get 'dfm_filename' parameter" );
    }

  } else {

    sprintf( log_file, "surf_%d.lis", srfno );
    warning_oldversion( );
    sprintf( msg, "logfile = %s", log_file );
    willbeassumed( msg );
    backcompatible_updateparfile( );

  }

  /*
*
*<=== ------------------------------------------------------------------
C
C        READ VALUE OF DFM_TYPE
C
*/
  if ( paccess( pfile, "dfm_type" ) ) {
    found_dfm_type = 1;
    dfm_type = pgeti( pfile, "dfm_type" );
  } else {
    found_dfm_type = 0;
    fprintf( stderr, "getpar.c: fix me please\n" );
  }

  /*    
C
C     IF THERE'S A SECOND DEFORMATION, GET DEFORMATION FILE NAME
C
*/
  fnm2_dfm = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == fnm2_dfm ) {
    free( fnm_gi );
    free( fnm_dfm );
    free( log_file );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'fnm_dfm'" );
  }
  fnm2_dfm[ 0 ] = '\0';
  if ( dfm_type >= 2 ) {
    if ( paccess( pfile, "dfm2_filename" ) ) {
      if ( NULL == pgetstr( pfile, "dfm2_filename", fnm2_dfm, SZ_PFLINE ) ) {
	free( fnm_gi );
	free( fnm_dfm );
	free( log_file );
	free( fnm2_dfm );
	tf_exit( ExitERR_IOerr, "Unable to get 'dfm2_filename' parameter" );
      }
    } else {
      tf_message( "surface_intercept: WARNING!\n"
		  "The second deformation file is missing!!!\n" );
    }
  }

  /*      
C
C   READ IN FIRST theta0
C
*/
  if ( paccess( pfile, "theta0" ) ) {
    theta0 = pgetd( pfile, "theta0" ) * deg2rad;
  } else {
    theta0 = 0.0;
    warning_oldversion( );
    strcpy( msg, "'theta0 = 0.0'" );
    willbeassumed( msg );
    backcompatible_updateparfile( );

  }

  /*
C
C     CALL FOR SECOND THETA (theta02) IF DFM_TYPE >=2
C
*/
  if ( dfm_type >= 2 ) {

    if ( paccess( pfile, "theta02" ) ) {
      theta02 = pgetd( pfile, "theta02" ) * deg2rad;
    } else {
      theta02 = 0.0;
      warning_oldversion( );
      strcpy( msg, "'theta02 = 0.0'" );
      willbeassumed( msg );
      backcompatible_updateparfile( );
    }

  }

  /*
C
C   READ IN FIRST scaling parameter
C
*/
  if ( paccess( pfile, "dfm_scale" ) ) {
    dfm_scale = pgetd( pfile, "dfm_scale" );
  } else {
    dfm_scale = 1.0;
    warning_oldversion( );
    strcpy( msg, "'dfm_scale = 1.0'" );
    willbeassumed( msg );
    backcompatible_updateparfile( );
  }

  /*
C
C    CALL FOR SECOND scaling parameter (dfm2_scale) IF DFM_TYPE>=2
C
*/
  if ( dfm_type >= 2 ) {

    if ( paccess( pfile, "dfm2_scale" ) ) {
      dfm2_scale = pgetd( pfile, "dfm2_scale" );
    } else {
      dfm2_scale = 1.0;
      warning_oldversion( );
      strcpy( msg, "'dfm2_scale = 1.0'" );
      willbeassumed( msg );
      backcompatible_updateparfile( );
    }

  }

  /*
C
C     call for Boolean flag for deformation
C
*/
  if ( paccess( pfile, "dfm_binary_io" ) ) {
    warning_oldversion( );
    tf_message( "surface_intercept: dfm_binary_io is no longer required.XS" );
  }

  if ( paccess( pfile, "do_osac_reflect" ) )
    do_osac_reflect = pgetb( pfile, "do_osac_reflect" );
  else {
    do_osac_reflect = 1;
    warning_oldversion( );
    strcpy( msg, "'do_osac_reflect = .TRUE.'" );
    willbeassumed( msg );
    tf_message( "surface_intercept: "
		"surface_intercept ***WILL*** reflect the rays\n" );
    backcompatible_updateparfile( );
  }
  
  debug = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == debug ) {
    free( fnm_gi );
    free( fnm_dfm );
    free( log_file );
    free( fnm2_dfm );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'fnm_dfm'" );
  }
  if ( NULL == pgetstr( pfile, "debug", debug, SZ_PFLINE ) ) {
    free( fnm_gi );
    free( fnm_dfm );
    free( log_file );
    free( fnm2_dfm );
    free( debug );
    tf_exit( ExitERR_IOerr, "Unable to get 'debug' parameter" );
  }

  if ( debug_init( debug ) )
  {
    tf_exit( ExitERR_IOerr, "out of memory in debug_init" );
  }

  *db_extend = dbflag( "extend_all" ) || dbflag( "save-history" );


  /*
  if ( pgetb( pfile, "help" ) ) {
    FUSAGE_F77( );
    paramclose( pfile );
    tf_leave( );
    exit( 0 );
  }
  */

  SETPARAM_F77( db_extend, &srfno, input, output, fnm_gi, fnm_dfm, log_file,
		&found_dfm_type, &dfm_type, fnm2_dfm, &theta0, &theta02,
		&dfm_scale, &dfm2_scale, &do_osac_reflect, debug,
		strlen( input ), strlen( output) , strlen( fnm_gi ),
		strlen( fnm_dfm ), strlen( log_file ), strlen( fnm2_dfm ),
		strlen( debug ) );

  free( fnm_gi );
  free( fnm_dfm );
  free( log_file );
  free( fnm2_dfm );
  free( debug );

}
