/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of src
 *
 * src is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * src is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 *       Free Software Foundation, Inc.
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <config.h>

#include <BsplineUtil/splines.h>
#include <bpipe/datatypes.h>
#include <cxcparam/parameter.h>
#include <tracefct/tracefct.h>
#include <tracefct/exiterrvals.h>
#include <saosacLib/osacbpipe.h>
#include <saosacLib/fproto.h>

#include "getpar.h"

#include "config.h"
#include "fproto.h"
#include "usage.h"

static
void loop_over_rays( void  ) {

  double* pos        = NULL;
  double* dir        = NULL;
  double* norm_c     = NULL;
  double* intersect_bcs_c = NULL;
  DComplex* c2comp_c = NULL;
  DComplex* s2comp_c = NULL;

  tf_enter( "loop_over_rays" );

  if ( NULL == (pos=(double*) malloc( 3 * sizeof( double ) )) )
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'pos'" );

  if ( NULL == (dir=(double*) malloc( 3 * sizeof( double ) )) ) {
    free( pos );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'dir'" );
  }

  if ( NULL == (norm_c=(double*) malloc( 3 * sizeof( double ) )) ) {
    free( pos );
    free( dir );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'dir'" );
  }

  if ( NULL == (intersect_bcs_c=(double*) malloc( 3 * sizeof( double ) )) ) {
    free( pos );
    free( dir );
    free( norm_c );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'dir'" );
  }

  if ( NULL == (c2comp_c=(DComplex*) malloc( 2 * sizeof( DComplex ) )) ) {
    free( pos );
    free( dir );
    free( norm_c );
    free( intersect_bcs_c );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'c2comp_c'" );
  }

  if ( NULL == (s2comp_c=(DComplex*) malloc( 2 * sizeof( DComplex ) )) ) {
    free( pos );
    free( dir );
    free( norm_c );
    free( intersect_bcs_c );
    free( c2comp_c );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 's2comp_c'" );
  }

  /*
   * Not really an infinite loop, because upon return
   * from SAONEXRAY: if 0 != ier then return to main.
   */
  for ( ;; ) {

    double w_c, energy_c, g_ang_c;
    int kray_c, kode_c, ray_missed, ier=1, outputray=0;

    /* get_ray_ is to be called from C instead of fortran SAONEXRAY */
    GET_RAY_F77( pos, dir, &w_c, &energy_c, c2comp_c, s2comp_c, &kray_c, &kode_c,
	      &ier );
    SAONEXRAY_F77( pos, dir, &w_c, &energy_c, c2comp_c, s2comp_c,
		   &kray_c, &kode_c, &ier );

    if ( 0 != ier ) {
      free( pos );
      free( dir );
      free( norm_c );
      free( intersect_bcs_c );
      free( c2comp_c );
      free( s2comp_c );
      tf_leave( );
      return;
    }

    SAOTRADER_F77( );

    SAOWRAD_F77( pos, dir, norm_c, intersect_bcs_c, &w_c, &energy_c,
		 &g_ang_c, c2comp_c, s2comp_c, &kray_c, &kode_c, &ray_missed,
		 &outputray );

    if ( outputray )
      PUT_RAY_F77( pos, dir, norm_c, intersect_bcs_c, &w_c, &energy_c, &g_ang_c,
		c2comp_c, s2comp_c, &kray_c, &kode_c, &ray_missed );

  }

  /* hopefully, we will never make it here */
  /* tf_leave( ); */

}


static
void mycleanup( int splinecleanup ) {

  tf_enter( "mycleanup" );

  /* we are we calling Fortran callable C routines here? */

  DETACH_BPIPE_F77( );
  FLUSHCLOSE8_F77( );
  if ( splinecleanup )
    SPLINE_CLEANUP_F77( );

  tf_leave( );
  return;

}

static
void myinit( int argc, char* argv[] ) {

  paramfile pfile = NULL;

  int db_extend;
  int srfno            = -1;
  int lsw0_srfno       = -1;
  int fgoodrays        = -1;
  char* input          = NULL;
  char* output         = NULL;

  tf_enter( "myinit" );

  /* open_paramfile will exit if pfile == NULL, no need to do it again here */
  pfile = open_paramfile( NULL, argc, argv, "r" );

  if ( pgetb( pfile, "version" ) ) {
    fprintf( stdout, "suface_intercept\t%s\n", VERSION );
    paramclose( pfile );
    tf_leave( );
    exit( 0 );
  }

  if ( pgetb( pfile, "help" ) ) {
    paramclose( pfile );
    usage();
    tf_leave( );
    exit( 0 );
  }

  srfno = pgeti( pfile, "surf_no" );

  input = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == input )
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'input'" );
  if ( NULL == pgetstr( pfile, "input", input, SZ_PFLINE ) ) {
    free( input );
    tf_exit( ExitERR_IOerr, "Unable to get 'input' parameter" );
  }

  output = (char*) malloc( SZ_PFLINE * sizeof( char ) );
  if ( NULL == output ) {
    free( input );
    tf_exit( ExitERR_alloc, "Unable to allocate memory for 'output'" );
  }
  if ( NULL == pgetstr( pfile, "output", output, SZ_PFLINE ) ) {
    free( input );
    free( output );
    tf_exit( ExitERR_IOerr, "Unable to get 'output' parameter" );
  }

  if ( paccess( pfile, "onlygoodrays" ) )
    fgoodrays = pgetb( pfile, "onlygoodrays" );
  else
    fgoodrays = 1;

  INITCONST_F77( &srfno, &lsw0_srfno );

  get_parameters( pfile, srfno, &db_extend, input, output );
  paramclose( pfile );

  /* Q: this is bizarre. attach_bpipe is a Fortran callable C routine
     why are we calling it from C?
     A: because we need to pass the fgoodrays flag?
  */
  ATTACH_BPIPE_F77( input, output, &db_extend, &fgoodrays, strlen( input ),
		 strlen( output ) );

  if ( INIT_SURFACE_INTERCEPT_F77( ) ) {
    mycleanup( 0 );
    tf_exit( ExitERR_IOerr, "INIT_SURFACE_INTERCEPT returned with an error" );
  }

  free( input );
  free( output );
  tf_leave( );

}

int main( int argc, char* argv[] ) {

  tf_init( argv[ 0 ], 0, -1 );

  tf_enter( "main" );

  myinit( argc, argv );

  loop_over_rays( );

  mycleanup( 1 );

  tf_leave( );

  return EXIT_SUCCESS;

}
