#! /usr/bin/env perl

use 5.012;

use Test2::V0;
use Readonly;

BEGIN {
    # pull this in early so we can use symbols from it during
    # compilation
    require 'trace-shell';
}

subtest 'boolean' => sub {

    Readonly::Array my @attr => qw( pcomm noexec noghosts );

    for my $test ( [ 'single', 0 ], [ 'double', 0 .. 1 ], [ 'triple', 0 .. 2 ] )
    {

        my ( $label, @idx ) = @$test;

        subtest $label => sub {

            my @attr = @attr[@idx];

            my $dbf;
            ok( lives { $dbf = DBFlags->new_from_string( join( ',', @attr ) ) },
                'construct' )
              or diag $@;

            for my $attr ( @attr ) {
                my $predicate = "has_$attr";
                ok( $dbf->$predicate, "$attr: set" );
                ok( $dbf->$attr,      "$attr: value" );
            }
        };
    }

};

subtest 'scalar' => sub {

    Readonly::Hash my %attr => (
        scat_max_prob   => 0.9,
        scat_min_prob   => 0.2,
        scat_max_prob_h => 0.8
    );

    Readonly::Array my @attr => keys %attr;

    for my $test ( [ 'single', 0 ], [ 'double', 0 .. 1 ], [ 'triple', 0 .. 2 ] )
    {

        my ( $label, @idx ) = @$test;

        subtest $label => sub {
            my @attr = @attr[@idx];
            my $dbf;
            ok(
                lives {
                    $dbf = DBFlags->new_from_string(
                        join( ',', map { "$_=$attr{$_}" } @attr ) )
                },
                'construct'
            ) or diag $@;

            for my $attr ( @attr ) {
                my $predicate = "has_$attr";
                ok( $dbf->$predicate, "$attr: set" );
                is( $dbf->$attr, $attr{$attr}, "$attr: value" );
            }
        };
    }
};

subtest 'hash' => sub {

    Readonly::Array my @attr      => qw( save-history save-rays save-history );
    Readonly::Array my @locations => DBFlags::HISTORY_LOCATIONS;

    for my $test ( [ 'single', 0 ], [ 'double', 0 .. 1 ], [ 'triple', 0 .. 2 ] )
    {

        my ( $label, @idx ) = @$test;

        subtest $label => sub {
            my @attr      = @attr[@idx];
            my @locations = @locations[@idx];

            my $dbf;
            ok(
                lives {
                    $dbf = DBFlags->new_from_string(
                        join( ',', map { "$attr[$_]:$locations[$_]" } @idx ) )
                },
                'construct'
            ) or diag $@;

            my %expected;
            ( $expected{ $attr[$_] } //= {} )->{ $locations[$_] } = 1 for @idx;

            for my $attr ( @attr ) {
                my $var       = $attr =~ s/-/_/gr;
                my $predicate = "has_${var}";
                ok( $dbf->$predicate, "$attr: set" );
                is( $dbf->$var, $expected{$attr}, "$attr: value" );
            }
        };
    }
};

subtest 'save_history_fields' => sub {
    my @fields = qw( a bc def );
    for my $test ( [ 'single', 0 ], [ 'double', 0 .. 1 ], [ 'triple', 0 .. 2 ] )
    {
        my ( $label, @idx ) = @$test;

        subtest $label => sub {
            my @fields = @fields[@idx];

            my $dbf;
            ok(
                lives {
                    $dbf = DBFlags->new_from_string(
                        'save-history-fields=' . join( ':', @fields ) );
                },
                'construct'
            ) or diag $@;

            ok( $dbf->has_save_history_fields, "set" );
            is( $dbf->save_history_fields, \@fields, "value" );
        };
    }
};

subtest 'complicated' => sub {

    my $dbf;
    ok(
        lives {
            $dbf = DBFlags->new_from_string(
                join( ',',
                    'pcomm',
                    'noexec',
                    'save-history-fields=a:b:c,scat_max_prob=0.9',
                    'noghosts',
                    'save-history:input',
                    'save-history:post-intercept-p',
                    'save-rays:post-intercept-h',
                    'save-history:pre-intercept-p',
                    'reuse',
                ) );
        },
        'construct'
    ) or diag $@;

    is(
        $dbf,
        object {
            call has_pcomm => T();
            call pcomm => T();
            call has_noexec => T();
            call noexec => T();
            call has_noghosts => T();
            call noghosts => T();
            call has_save_history_fields => T();
            call save_history_fields => [ qw( a b c ) ];
            call save_history => { map { $_ => 1 }  qw( input post-intercept-p pre-intercept-p ) };
            call save_rays => { map { $_ => 1 }  qw( post-intercept-h ) };
        },
       'values'
    );
};

done_testing;
