package My::Test::Util;

use strict;
use warnings;

use Exporter qw( import );
use Storable qw( dclone );
use Data::Rmap qw( rmap );
use File::Spec::Functions qw( catdir catfile );
use Scalar::Util qw( looks_like_number );
use File::Slurper qw( read_lines );
use YAML;
use Regexp::Common qw( number );

use Test2::Tools::Compare qw( within bag item end array hash field );

use constant TOLERANCE => 1e-15;

BEGIN {
    # this gives us BP2ITS_CONFIG_DIR
    require 'trace-shell';
}

#<<< no tidy
use constant SAOTRACE_DB  => $ENV{SAOTRACE_DB};
use constant DB_ROOT      => SAOTRACE_DB;
use constant TESTDIR      => $ENV{ABS_TESTDIR} // die( "missing ABS_TESTDIR environment variable" );
use constant EXPECTED_DIR => catdir( TESTDIR, 'expected' );
use constant BUILDDIR     => $ENV{ABS_TOP_BUILDDIR};
#>>> no tidy

our @EXPORT = qw(
  load_exp_yaml save_exp_yaml
  read_exp_lines
  read_lines
  load_gi load_exp_gi
  load_rdb load_exp_rdb
  DB_ROOT TESTDIR BP2FITS_CONFIG_DIR
  SAOTRACE_DB EXPECTED_DIR BUILDDIR
);

sub expfile {
    catfile( EXPECTED_DIR, @_ );
}

sub save_exp_yaml {
    my ( $data, @path ) = @_;
    save_yaml( expfile(  @path ), $data );
}

sub load_exp_yaml {
    my ( @path ) = @_;
    load_yaml( expfile( @path ) );
}

sub read_exp_lines {
    my ( @path ) = @_;
    [
        map( &_interpolate_exp,
            File::Slurper::read_lines( expfile( @path ) ) ) ];
}

sub _uninterpolate_exp {
    unless ( looks_like_number( $_ ) ) {
        s/@{[ DB_ROOT ]}/DB_ROOT/g;
        s/@{[ SAOTRACE_DB ]}/SAOTRACE_DB/g
          if defined SAOTRACE_DB;
        s/@{[ BP2FITS_CONFIG_DIR ]}/BP2FITS_CONFIG_DIR/g;
    }
    return $_;
}

sub _interpolate_exp {
    if ( looks_like_number( $_ ) ) {
        return within( $_, TOLERANCE );
    }
    else {
        s/DB_ROOT/@{[ DB_ROOT ]}/g;
        s/BP2FITS_CONFIG_DIR/@{[ BP2FITS_CONFIG_DIR ]}/g;
        s/SAOTRACE_DB/@{[ SAOTRACE_DB ]}/g
          if defined SAOTRACE_DB;
    }
    return $_;
}

sub save_yaml {
    my ( $file, $data ) = @_;
    my $clone = dclone( $data );
    rmap( \&_uninterpolate_exp, $clone );
    YAML::DumpFile( $file, $clone );
}

sub load_yaml {
    my ( $file ) = @_;
    my $fiducial = YAML::LoadFile( $file );
    rmap( \&_interpolate_exp, $fiducial );
    return $fiducial;
}

sub load_gi {
    my ( $file ) = @_;
    my @got;
    for my $rec ( read_lines( $file ) ) {
        if ( $rec =~ /^tag/ ) {
            push @got, $rec;
        }
        else {
            my ( $shell, $id, $value, $comment )
              = $rec =~ /^($RE{num}{int}),($RE{num}{int}),($RE{num}{real})\s*(.*)$/
              or die( "unable to parse GI record: $rec" );
            push @got, [ $shell, $id, $value, $comment ];
        }
    }
    return \@got;
}

sub load_exp_gi {
    my $exp = load_gi( expfile( @_ ) );
    return bag {
        for my $rec ( @$exp ) {
            if ( 'ARRAY' eq ref $rec ) {
                $rec->[$_] = within( $rec->[$_], TOLERANCE ) for 0, 1, 2;
                item array { item $_ foreach @$rec; end; };
            }
            else {
                item $rec;
            }
        }
        end();
    }
}

sub load_rdb {
    my ( $file ) = @_;
    my @rows;
    my @header;
    my $ncols;
    for my $rec ( read_lines( $file ) ) {
        next if $rec =~ /^#/;
        if ( !@header ) {
            chomp $rec;
            push @header, split(/\t/, $rec );
            $ncols = @header;
            read_lines( $file );
        }
        else {
            my %row;
            @row{@header} = split(/\t/, $rec, $ncols );
            push @rows, \%row;
        }
    }
    return \@rows;
}

sub load_exp_rdb {
    my $rows = load_rdb( expfile( @_ ) );
    return array {
        for my $row ( @$rows ) {
            item hash {
                while( my ( $key, $value ) = each %$row ) {
                    # use less tolerance here to get it to pass on MacOS/clang
                    field $key, looks_like_number($value) ? within( $value, 1e-9 ) : $value;
                }
                end;
            }
        }
        end();
    }
}

1;
