#! /usr/bin/env perl

use 5.012;

use Test2::V0;
use CIAO::Lib::Param;
use Test::TempDir::Tiny;

use Readonly;

use Data::Dumper;
use File::Spec::Functions qw( catfile catdir );

use My::Test::Param;
use My::Test::Util;

require 'trace-shell';

sub fiducial_file {
    my ( $label, $shell ) = @_;
    $label =~ s/\W/_/g;
    return join( '.', $label, $shell, 'yml' );
}

sub save_fiducial {
    my ( $fiducial, $label, $shell ) = @_;
    save_exp_yaml( $fiducial, 'pipes', fiducial_file( $label, $shell ) );
}

sub load_fiducial_check {
    my ( $label, $shell ) = @_;
    my $fiducial = load_exp_yaml( 'pipes', fiducial_file( $label, $shell ) );

    hash {
        while ( my ( $key, $value ) = each %{$fiducial} ) {
            if ( $key eq 'elements' ) {
                field elements => array {
                    while ( my ( $idx, $element ) = each @{$value} ) {
                        item hash {
                            while ( my ( $key, $value ) = each %{$element} ) {
                                if ( $key eq 'cmd' ) {
                                    field( cmd => $value );
                                }
                                elsif ( $key eq 'args' ) {
                                    field(
                                        args => bag {
                                            item( $_ ) for @{$value};
                                            end;
                                        } );
                                }
                                else {
                                    die "element[$idx]: unknown key: $key\n";
                                }
                            }
                        };
                        end;
                    }
                };
            }
            else {
                die "unknown top level key: $key\n";
            }
        }
        end;
    };
}

Readonly::Hash my %PARAM => (
    block         => 0,
    block_inc     => 100,
    config_db     => "orbit-200809-01f-a.cnf",
    config_dir    => catdir( DB_ROOT, "ts_config" ),
    debug         => DBFlags->new(),
    focus         => 0,
    help          => "",
    limit         => 0.1,
    limit_type    => "r/mm2",
    mode          => "a",
    output        => "default",
    output_coord  => "hrma",
    output_fields => "all",
    output_fmt    => "bpipe",
    seed1         => 1,
    seed2         => 1,
    src           => "default",
    srcpars       => "spectrum=1.49",
    tag           => "foo",
    tally         => 0,
    tstart        => 0,
    version       => "",
    z             => 193.925,
    bp2fits_config_dir       => BP2FITS_CONFIG_DIR,
);


Readonly::Hash my %Tests => (

    default => {},

    focus => {
        params => { focus => 1 }
    },

    tally => {
        params => { tally => 1000 },
    },

    # output formats
    (
        map { ; "output format: $_" => { params => { output_fmt => $_ } } }
          qw( bp fr rdb fits-events)
    ),

    # output coords
    (
        map { ; "output coord: $_" => { params => { output_coord => $_ } } }
          qw( xrcf osac hrma )
    ),

    # output fields
    (
        map { ; "output field: $_" => { params => { output_fields => $_ } } }
            ('all', 'min', 'position,direction,weight')
    ),

    'normalize limit' => {
        params => { debug => DBFlags->new( normalize_limit => 1 ) }
    },

    'noproject' => {
        params => { debug => DBFlags->new( noproject => 1 ) }
    },

    'noghosts' => {
        params => { debug => DBFlags->new( noghosts => 1 ) }
    },

    (
        map {
            ;
            my $attr = $_;
            map {
                my $optic = $_;
                my $sys_var = join( '_', $attr, ( $optic // () ) );
                "${sys_var} = 0" =>
                  { params => { debug => DBFlags->new( $sys_var => 0 ) } };
            } undef, 'h', 'p';
        } qw( scat_in_plane scat_out_of_plane )
    ),

    (
        map {
            ;
            my ( $attr, $value ) = @{$_};
            map {
                my $optic = $_;
                my $sys_var = join( '_', $attr, ( $optic // () ) );
                "${sys_var} = $value" =>
                  { params => { debug => DBFlags->new( $sys_var => $value ) } };
            } undef, 'h', 'p';
        }[ scat_min_prob => 0.2 ],
        [ scat_max_prob  => 0.8 ]
    ),

    (
        do {
            my %locations = (
                input => 1,
                map { ; "$_-p" => 1, "$_-h" => 1 }
                  qw( pre-scatter post-scatter pre-intercept post-intercept pre-reflect post-reflect )
            );
            map {
                $_ => {
                    params => {
                        debug => DBFlags->new(
                            $_ => \%locations,
                            (
                                $_ eq 'save-history'
                                ? ( 'save-history-fields' =>
                                      [qw( position direction id)], )
                                : (),
                            )
                        ),
                    },
                };
            } 'save-history', 'save-rays';
          }
    ),
);

while ( my ( $label, $args ) = each %Tests ) {

    my @shells = @{ $args->{shells} // [ 1, 3, 4, 6 ] };
    my %test_params = %{ $args->{params} // {} };

    subtest $label => sub {

        for my $shell ( @shells ) {

            in_tempdir "${label}.${shell}" => sub {
                my %param = ( %PARAM, shell => $shell, %test_params );

                my $trace;
                ok( lives{ $trace = TraceShell->new( %param ) }, "create raytrace" )
                  or do {
                      diag $@;
                      diag( Data::Dumper->Dump( [ \%param ] ) );
                  };

                my $pipe;
                ok( lives { $pipe = $trace->raytrace_pipeline }, "create pipeline" )
                  or do {
                      diag $@;
                      diag( Data::Dumper->Dump( [ \%param ] ) );
                  };
                $pipe->renderer( 'Struct' );

                # save_fiducial( $pipe->render, $label, $shell )
                #   if $args->{save};
                my $check = load_fiducial_check( $label, $shell );

                my $output;
                ok( lives { $output = $pipe->render }, "render pipe" )
                  or diag( Data::Dumper->Dump( [ \%param ] ) );

                is( $output, $check, "shell $shell" )
                  or do {
                    diag( Data::Dumper->Dump( [ \%param ] ) );
                    diag( Data::Dumper->Dump( [ $output ] ) );
                  };
            };
        }
    };
}

done_testing;
