import os
import pickle
from pathlib import Path

import pytest
from pkg_resources import resource_filename

from .io.adapters.crates import CratesInputOutputFactory


@pytest.fixture
def io_factory():
    return CratesInputOutputFactory()


@pytest.fixture
def deserialize(request):
    def deserialize_function(*relative_path):
        filename = resource_filename(request.module.__name__, os.path.join('', *relative_path))
        with open(filename, 'rb') as f:
            return pickle.load(f)

    return deserialize_function


@pytest.fixture
def make_data_path(request):
    def wrapper(*relative_path):
        filename = resource_filename(request.module.__name__, os.path.join('', *relative_path))
        return Path(filename)

    return wrapper


@pytest.fixture
def make_input_path():
    base_dir = '/data/L3/unit_tests/indir'

    def input_factory_function(*args):
        return str(Path(base_dir, *args))

    return input_factory_function


@pytest.fixture
def make_output_path():
    base_dir= "{0}".format(os.environ["TESTOUT"])

    # clean out the directory and remove
    def delete_dir(_path):
        for sub in _path.iterdir():
            if sub.is_dir():
                # Delete sub directories recursively
                delete_dir(sub)
            else :
                # Delete file
                sub.unlink()

        # This removes the top-level folder:
        _path.rmdir()

    def output_factory_function(*args):
        out_dir = Path(base_dir, *args)
        if out_dir.exists():
            delete_dir(out_dir)
        out_dir.mkdir(parents=True, exist_ok=True)
        return str(out_dir)

    return output_factory_function


@pytest.fixture
def make_savdir_path():
    base_dir = '/data/L3/unit_tests/savdir'

    def savdir_factory_function(*args):
        return str(Path(base_dir, *args))

    return savdir_factory_function


@pytest.fixture
def count_outfiles():
    base_dir= "{0}".format(os.environ["TESTOUT"])
    def count_outfiles_factory_function(*args, extension='fits'):
        my_dir = Path(base_dir, *args)
        files = my_dir.glob(f'*.{extension}')
        return len([str(file) for file in files])

    return count_outfiles_factory_function


@pytest.fixture
def make_mrgsrc3_files_list():
    base_dir = '/data/L3/unit_tests/indir/'

    def input_factory_function(*args, extension='fits'):
        my_dir = Path(base_dir, *args)
        files = my_dir.glob(f'*_mrgsrc3*.{extension}')
        filenames = [str(file) for file in files]
        return ','.join(filenames)

    return input_factory_function


@pytest.fixture
def make_prop3_files_list():
    base_dir = '/data/L3/unit_tests/indir/'

    def input_factory_function(*args, extension='fits'):
        dir = Path(base_dir, *args)
        files = dir.glob(f'*_prop3*.{extension}')
        filenames = [str(file) for file in files]
        return ','.join(filenames)

    return input_factory_function


def pytest_addoption(parser):
    parser.addoption(
        "--no-mock-pymc", action="store_true", default=False, help="do not mock pymc3 (tests will be slower)"
    )
