
# 
#  Copyright (C) 2011-2013,2016,2020-2021  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#


from __future__ import print_function

import unittest
import sys,os,re
from pydoc import help   # for run-time help
from caldb4 import *     # the PACKAGE

import subprocess  



def show_whoami():
    print ('\n---', sys._getframe(1).f_code.co_name)
    sys.stdout.flush()

def usage():
    """Show the usage with example for Caldb python module."""
    
    print ("""\nUsage:\n\
           % python test_caldb4.py [opt] [-v]
           opt:
             -t1:   test the normal procedure
             -t2:   test the ways of get 'allfile'
             -t3:   test resetting 'product' and moving on
             -t4:   test setting telescope and instrument
             -t5:   test setting unrelevant attributes
             -t6:   test updating 'product'
             -t7:   test updating 'infile' option
             -t9:   test environs properly setups 
             -t11:  test CALDB-lib crash fixing
             -u:   this message\n
             -v:   debug option\n""")

    sys.stdout.flush()

def _split_path_and_file(afile, delim):
    p,f=0,0
    delim.strip()
    p,f=re.split(delim, afile, flags=re.IGNORECASE)            
    f.strip()  # remove white space
    return p,f
    

 
DEBUG=0
class TestCaldb4lib(unittest.TestCase):
    def setUp(self):
        global DEBUG
        if "-v" in  sys.argv:
            DEBUG=1
            #sys.argv.remove("-v")
            
    def tearDown(self):
        """ un-import caldb4 ?"""
        nm=['caldb4', '']
        while True:
            if nm[0] in sys.modules:
                del (sys.modules[nm[0]])
                del (nm[0])
            else:
                break
        return
#=============================================================================
#
#   --- The Tester ---
#
#=============================================================================

#=======================================
# test 1 - test the normal procedure
#========================================
    def test1(self):
        try:
            if DEBUG: show_whoami()
            c = Caldb("chandra", "acis", "badpix")
            if DEBUG: print (c)
  
            #c.start="2000-01-29T20:00:00"
            c.start="2000-11-28"

            c.fp_temp=158.0,"K"
  
            c.grating ='None'
            if DEBUG: print (c)

            files=c.search
            num=0
            if files != None:
                print ("\nnum files: ",  len(files))
                ii=1
                for f in files:
                    print (ii,"> ", os.path.basename(f) )
                    ii += 1

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)


#=======================================
# test 2 - test the ways of get 'calfile'
#========================================
    def test2(self):

        try:
            if DEBUG: show_whoami()
            c = Caldb("chandra", "acis")
            if DEBUG: print (c)
  
            c.product="badpix"
            #c.start="2000-01-29T20:00:00"
            c.start="2000-11-28"
            if DEBUG: print (c)

            myfiles=c.search
            num=0
            if myfiles != None:
                num=len(myfiles)
                print ("\nnum files: ", num)
            for ii in range(1,num+1):
                print (ii,"> ", os.path.basename(myfiles[ii-1]))

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)

#=======================================
# test 3 - reset product and moving on
#========================================
    def test3(self):

        try:
            if DEBUG: show_whoami()
            c = Caldb("chandra", "acis", "badpix")
            if DEBUG: print (c)


            #c.start="2000-01-29T20:00:00"
            c.start="2000-11-28"
            if DEBUG: print (c)

            files=c.search
            num=len(files)
            print ("num files: ", num)
            for ii in range(1,num+1):
                print (ii,"> ", os.path.basename(files[ii-1]))

            c.product="det_gain"
            if DEBUG: print (c)

            files=c.search
            num=len(files)
            print ("num files: ", num)
            for ii in range(1,num+1):
                print (ii,"> ", os.path.basename(files[ii-1]))

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)


#======================================================
# test 4 - telescope and instrument are not updatable
#======================================================
    def test4(self):

        try:
            if DEBUG: show_whoami()
            c = Caldb("chandra", "acis","det_gain")
            if DEBUG:
                print ("My curr inputs/needs: ", c)
                print ("\nset 'telescope' and 'instrument' with new values")
            c.telescope="newteless"
            c.instrume="newinstrr"
            if DEBUG: print ("My curr inputs/needs: ", c)
            files=c.search
            num=len(files)
            print ("num files: ", num)
            for ii in range(1,num+1):
                print (ii,"> ", os.path.basename(files[ii-1]))


            if DEBUG:
                print ("\nset 'tele' and 'inst' with new values")
                c.tele="newtele"
                c.inst="newinst"
                print ("My curr inputs/needs: ", c)

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)

#=======================================
# test 5 - set unrelevant attributes
#========================================
    def test5(self):

        try:
            if DEBUG: show_whoami()
            c = caldb4.Caldb("chandra", "acis")
            if DEBUG: print ( c)

            c.product="badpix"
            #c.start="2000-01-29T20:00:00"
            c.start="2000-11-28"
            c.unknown = 'gabage'
            if DEBUG: print (c)

            files=c.search
            num=len(files)
            if num >=3:
                num=3

            print ("num files: ", num)
            for ii in range(1,num+1):
                print (ii,"> ", os.path.basename( files[ii-1]))

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)

#======================================================
# test 6 - sync test of the updated product
#======================================================
    def test6(self):


        try:
            if DEBUG: show_whoami()
            c = Caldb("chandra", "acis","t_gain")
            if DEBUG: print ("My curr inputs/needs:\n", c)
            files=c.search
            nfiles=len(files)
            print ("found ", nfiles, " files, the 1st file:")
            print (os.path.basename(files[0]), "\n")

            c.product="bkgrnd"
            if DEBUG:
                print ("Updating product to ", repr(c.product))
                print ("My curr inputs/needs:\n", c)
            files=c.search
            nfiles=len(files)
            if DEBUG:
                print ("expected ", nfiles, " files found for the inputs above.")

            c.CTI_APP='NNNNNNNNNN'
            c.SIM_Z= '-233.5874344608','mm' #-227.7139939164
            c.fp_temp='148.2-158.2','K'
            c.CCD_ID=3
            if DEBUG: print ("My curr inputs/needs:\n", c)
            cfiles=c.search
            nn=len(files)
            if DEBUG: print ("found ", nn, " files:")
            exp1= 'acis3iD2000-01-29bkgrndN0003.fits'
            exp2='acis3iD2000-12-01bkgrndN0002.fits'

            for f in files:
                if f.find(exp1) != -1 or f.find(exp2) != -1:
                    print ("expected: ",os.path.basename(f))
            if DEBUG: print ("-------------\n")

            del c.sim_z
            del c.fp_temp
            del c.cti_app
            del c.ccd_id
            if DEBUG:
                print ("Delete sim_z, fp_temp, cti_app, and ccd_id.")
                print ("My curr inputs/needs: ", c)
            files = c.search
            nfiles=len(files)
            if nfiles == nn:
                print ("expected: ", nfiles, " files found.")
            else:
                print ("unexpected: ", nn, " files found compared to ",nfiles,".")
            if DEBUG: print ("-------------\n")

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)

#======================================================
# test 7 - test 'infile' auto-querying
#======================================================
    def test7(self):

        try:
            if DEBUG: show_whoami()
            thepath = os.getenv("TESTIN")
    
            thepath += "/calquiz"
            file="obs_3399_evt1.fits"
            infile= thepath +'/'+file
            if not os.path.exists(infile):
                #skip the testing
                print ("\ninfile '",infile,"', does not exists. Skip the test.")
                return
           

            c = Caldb("","","GREFF", infile)
            c.grating='letg'
            c.shell='0100'

            files=c.search
      
            nfiles=0
            if not files:
                print ("Error: un-expected: ", nfiles, " files found via infile-auto-querying.")
                return

            nfiles=len(files)
            nn   = 1          # expected
            exp1 = "letgD1996-11-01greffpr001N0005.fits"
            if nfiles==nn and files[0].find(exp1) != -1:
                print ("expected: ", nfiles, " files found via infile-auto-querying.")
            else:
                if nfiles!=nn:    
                    print ("unexpected: ", nfiles, " files found compared to ",nn,".")
                else:
                    print ("unexpected: '", file[0], "' found compared to '",exp1,"'.")
        
            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)

#===========================================================
# test 8 - test default arguments: tele, inst, prod,infile
#=========================================================
    def test8(self):

        try:
            c = Caldb(None)            # exit error
        except caldb4.caldbError as err:
            print(err.value)

        try:
            c = Caldb("")              # exit error
        except caldb4.caldbError as err:
            print (err.value)

        try:
            c = Caldb(0)               # exit error
        except caldb4.caldbError as err:
            print(err.value)

        try:
            c = Caldb(instrume=None)   # True == instrueme="", and others to defaults
            c.close
        except caldb4.caldbError as err:
            print(err.value)

        try:
            c = Caldb(infile=None)     # True == infile="",    and others to defaults
            c.close
        except caldb4.caldbError as err:
            print(err.value)

        try:
            c = Caldb()                # all to defaults
            c.close
        except caldb4.caldbError as err:
            print(err.value)

#============================================================================
# test 9 - test MUST-HAVE environment variables not set, part of ticket SL210
#============================================================================
    def test9(self):


        # reserve the values of the keys
        try:
            caldbpath   = os.environ['CALDB']
            caldbconfig = os.environ['CALDBCONFIG']
            caldbalias  = os.environ['CALDBALIAS']
        except KeyError as exc:
            print ("KeyError: {}.".format(str(exc)))
            return 0
 

        try:
            # delete the key CALDB
            del os.environ['CALDB'] 
            c = Caldb("chandra",  None, "msidmap")
            c.close
        except caldb4.caldbError as e:
            print(e.value) 
        except Exception as exc:
            print(exc)
        # restore CALDB
        os.environ['CALDB'] = caldbpath


        try:
            # delete key CALDBCONFIG
            del os.environ['CALDBCONFIG'] 
            c = Caldb("chandra",  None, "msidmap")
            c.close
        except caldb4.caldbError as e:
            print(e.value) 
        except Exception as exc:
            print(exc)
        # restore CALDBCONFIG
        os.environ['CALDBCONFIG'] = caldbconfig        

        try:
            # delete key CALDBALIAS
            del os.environ['CALDBALIAS'] 
            c = Caldb("chandra",  None, "msidmap")
            c.close
        except caldb4.caldbError as e:
            print(e.value) 
        except Exception as exc:
            print(exc)
        # restore CALDBALIAS
        os.environ['CALDBALIAS'] = caldbalias
  
            

        #  telescope, instrume,  product
        try:
            # change or empty the value of Key CALDB
            os.environ['CALDB'] = ""
            c = Caldb("chandra",  None, "msidmap")
            c.close
        except caldb4.caldbError as exc:
            print(exc.value) 
        except Exception as exc:
            print(exc)
        # restore CALDB
        os.environ['CALDB'] = caldbpath


        try:
            c=Caldb(instrume="hrc", product="tgmask2")
            myfiles=c.search
            f=myfiles
            for afile in myfiles:
                f=afile
                if afile.lower().find("caldb4/")  !=-1:
                    p,f = _split_path_and_file(afile,"caldb4/")
                elif afile.lower().find("caldb/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb/")

            c.close
            print( "<CALDB>/"+f)  # expect <CALDB>/data/chandra/hrc/tgmask2/letgD1999-07-22regN0002.fits[1]


            c=Caldb(instrume="",product="tgmask2")
            myfiles=c.search
            f=0
            for afile in myfiles:
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    # do nothing now
                    f=afile
            print( "<CALDB>/"+f)  # expect <CALDB>/data/chandra/hrc/tgmask2/letgD1999-07-22regN0002.fits[1]
     
            c=Caldb(instrume=None,product="tgmask2")
            myfiles=c.search
            for afile in myfiles:
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    # do nothing now
                    f=afile
            print( "<CALDB>/"+f)  # expect <CALDB>/data/chandra/hrc/tgmask2/letgD1999-07-22regN0002.fits[1]
 
            c=Caldb(instrume=0,product="tgmask2")
            myfiles=c.search
            for afile in myfiles:
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    # do nothing now
                    f=afile
            print( "<CALDB>/"+f)  # expect <CALDB>/data/chandra/hrc/tgmask2/letgD1999-07-22regN0002.fits[1]
 
            c=Caldb(product="tgmask2", instrume=None)
            myfiles=c.search
            f=0
            for afile in myfiles:
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    # do nothing now
                    f=afile
            print( "<CALDB>/"+f)  # expect <CALDB>/data/chandra/hrc/tgmask2/letgD1999-07-22regN0002.fits[1]
 

            # expected nothing return as 'instrume=default'
            c=Caldb(product="tgmask2")
            myfiles=c.search
            f="[]"
            for afile in myfiles:
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    # do nothing now
                    f=afile
            print( "<CALDB>/"+f)  # expect <CALDB>/[]
 


            c=Caldb()
            myfiles=c.search  # expected error

            c.close

        except caldb4.caldbError as exc:
            print(exc.value)
        except Exception as exc:
            print(exc)

#===========================================================
# test 10 - test CALDB string types
#=========================================================
    def test10(self):

        try:
         #1 
            print("\n-- Testing all types of strings")
            c=Caldb( b"chandra", u"acis", None, "")
            print('tele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, ', infile=', c.infile)
            print(c.search)
            print(c.errmsg)
            c.close
        except caldb4.caldbError as exc:
            print( exc.value )
        except Exception as exc:
            print(exc)

        try:
         #2 
            print("\n-- Testing inputs with defaults input")
            c=Caldb() 
            c.product="badpix"
            print('tele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            print(c.search)
            print(c.errmsg)
            c.close
        except caldb4.caldbError as exc:
            print( exc.value )
        except Exception as exc:
            print(exc)

        try:
         #3 
            print("\n-- Testing product='' ")
            c=Caldb( "chandra", "acis", b"")
            print('tele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            print(c.search)
            print(c.errmsg)
            c.close

        except caldb4.caldbError as exc:
            print( exc.value )
        except Exception as exc:
            print(exc)
    
#===========================================================
# test 11 - test c-CALDB-lib crash fixing
#=========================================================
    def test11(self):

        try:

            # start new queries with caldb-initiation
            c=Caldb(instrume=b"", product="badpix")
            print('tele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            print(c.allneeded)
            myfiles=c.search
            if myfiles:
                afile=myfiles[0]
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    f=afile
                print("#founds: ", len(myfiles), " 1st file: ",f)

            # start new queries under the same caldb-initiation
            c.detnam= b"hrc-i"
            c.fp_temp=158, b"k"
            myfiles=c.search            
            print('\ntele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",(c.detnam).decode())
            print(c.allneeded)
            if myfiles:
                afile=myfiles[0]
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    f=afile
                print("#founds: ", len(myfiles), " 1st file: ",f)
                       
  
            # start the 3rd new queries under the same caldb-initiation
            del c.fp_temp
            c.detnam=u"acis"
            myfiles=c.search
            print('\ntele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            if myfiles:
                afile=myfiles[0]
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    f=afile
                print("#founds: ", len(myfiles), " 1st file: ",f)


            # start the 4th new queries under the same caldb-initiation
            c.detnam="hrc-s"
            myfiles=c.search
            print('\ntele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            if myfiles: 
                afile=myfiles[0]
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    f=afile
                print("#founds: ", len(myfiles), " 1st file: ",f)


            # start the 5th new queries under the same caldb-initiation
            # test set-instrume is no-effect on the search, as the 'instrume' is  retained as whatever in Caldb() initiation 
            #  set-instrume with 'acis'
            del c.detnam
            c.instrume="acis"
            myfiles=c.search
            print('\ntele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            if len(myfiles):
                afile=myfiles[0]
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    f=afile
                print("#founds: ", len(myfiles), " 1st file: ",f)

            # start the 6th new queries under the same caldb-initiation
            # set-instrume with 'default'
            c.instrume="default" 
            myfiles=c.search
            print('\ntele=',c.telescope, ', instru=', c.instrume, ', prod=', c.product, " detnam=",c.detnam)
            if len(myfiles):
                afile=myfiles[0]
                if afile.lower().find("caldb/") !=-1:
                    p,f = _split_path_and_file(afile,"caldb/")
                elif afile.lower().find("caldb4/") !=-1:
                    p,f=_split_path_and_file(afile, "caldb4/")
                else:
                    f=afile
                print("#founds: ", len(myfiles), " 1st file: ",f)

            c.close

        except caldb4.caldbError as exe:
            print( exe.value )
        except Exception as exc:
            print(exc)

#===========================================================
# test 12 - test infile formats,  part of ticket SL210
#=========================================================
    def test12(self):

        #------
        print("\n --- DM-FITS FILES ---")
        #-------

        testin = os.getenv("TESTIN")
  

        myinput=testin+'/caldb4/acisf09697_000N001_evt1.fits'
        try:
            print('\n01> myinput={}'.format(str(myinput)))
            print('    CMD:  cdb = caldb4.Caldb(infile=myinput)')

            cdb = caldb4.Caldb(infile=myinput)
        except Exception as exc:
            print("Exception: {}".format(str(exc)))

        try:
            print('\n02> CMD:  ftr_input=myinput+"[GTI6]")')
            print('    CMD:  cdb = caldb4.Caldb(infile=ftr_input)')
            ftr_input=myinput+"[GTI6]"
            cdb = caldb4.Caldb(infile=ftr_input)
        except Exception as exc:
            print("Exception: {}".format(str(exc)))

        try:
            print('\n03> CMD:  ftr_input=myinput+"[cols pha,sky]")')
            print('    CMD:  cdb = caldb4.Caldb(infile=ftr_input)')
            ftr_input=myinput+"[cols pha,sky]"
            cdb = caldb4.Caldb(infile=ftr_input)
        except Exception as exc:
            print("Exception: {}".format(str(exc)))

        try:
            print('\n04> CMD:  ftr_input=myinput+"[cols time,energy][time=318798730:318798739, energy=100:155]")')
            print('    CMD:  cdb = caldb4.Caldb(infile=ftr_input)')
            ftr_input=myinput+"[cols time,energy][time=318798730:318798739, energy=100:155]"
            cdb = caldb4.Caldb(infile=ftr_input)
        except Exception as exc:
            print("Exception: {}\n".format(str(exc)))


        #------
        print("\n --- PAR FILES ---")
        #-------

        try:
            print('\n05-1> CMD:  par_input="geom.par"')
            print('      CMD:  cdb = caldb4.Caldb(infile=par_input)')
            par_input='geom.par'
            cdb = caldb4.Caldb(infile=par_input)
        except Exception as exc:
            print("Exception: {}".format(str(exc)))

        try:
            print('\n05-2> CMD:  par_input="geom.par"')
            print('      CMD:  cdb = caldb4.Caldb(None,None, "foo",infile=par_input)')
            print('      CMD:  cdb.search')
            par_input='geom.par'
            cdb = caldb4.Caldb(None,None,"foo",infile=par_input)
            cdb.search
        except Exception as exc:
            print("Exception: {}\n".format(str(exc)))

        try:
            print('\n06-1> CMD:  par_input="geom"')
            print('      CMD:  cdb = caldb4.Caldb(infile=par_input)')
            par_input='geom'
            cdb = caldb4.Caldb(infile=par_input)
        except Exception as exc:
            print("Exception: {}\n".format(str(exc)))

        try:
            print('\n06-2> CMD:  par_input="geom"')
            print('      CMD:  cdb = caldb4.Caldb(None,None,"foo",infile=par_input)')
            print('      CMD:  cdb.search')
            par_input='geom'
            cdb = caldb4.Caldb(None,None, "foo", infile=par_input)
            cdb.search
        except Exception as exc:
            print("Exception: {}\n".format(str(exc)))

        #--------
        print("\n --- URL-LOCAL FILES ---\n")
        #--------

        myinput=testin+'/calquiz/obs_3399_evt1.fits'

        try:
            url_input='file://'+myinput
            print('\n07> CMD:  url_input={}'.format(str(url_input)))
            print('    CMD:  cdb = caldb4.Caldb("","","GREFF",infile=url_input)')
            print('    CMD:  cdb.serach')
            cdb = caldb4.Caldb("","","GREFF",infile=url_input)
            cdb.search
        except Exception as exc:
            print("Exception: {}".format(str(exc)))

        try:
            url_input='file:'+myinput
            print('\n08> CMD:  url_input={}'.format(str(url_input)))
            print('    CMD:  cdb = caldb4.Caldb("","","GREFF",infile=url_input)')
            print('    CMD:  cdb.serach')

            cdb = caldb4.Caldb("","","GREFF",infile=url_input)
            cdb.search
        except Exception as exc:
            print("Exception: {}".format(str(exc)))

        try:
            url_input='file://cxc.cfa.harvard.edu/'+myinput
            print('\n09> CMD:  url_input={}'.format(str(url_input)))
            print('    CMD:  cdb = caldb4.Caldb("","","GREFF",infile=url_input)')
            print('    CMD:  cdb.serach')
            cdb = caldb4.Caldb("","","GREFF",infile=url_input)
            cdb.search
        except Exception as exc:
            print("Exception: {}\n".format(str(exc)))


        #--------
        print("\n --- URL-REMOTE FILES ---\n")
        #--------


        try:
            url_input='https://cxc.cfa.harvard.edu/ciao/ahelp/ciao-install.html'
            print('\n10> CMD:  url_input={}'.format(str(url_input)))
            print('    CMD:  cdb = caldb4.Caldb("","","foo",infile=url_input)')
            cdb = caldb4.Caldb("","","foo",infile=url_input)
        except Exception as exc:
            print("Exception: {}\n".format(str(exc)))


        #--------
        print("\n")
        #--------
        
#=========================================================================================
# test 13  
#   Vary TimeZones in UTC or non-UTC under STD and DST conditions.
#   Do time querings, one start-time in DST months (4-10) and other in STD months (12-2).
#   Expect the quried products identical in both DST or STD months.
#  
# Self-verified test unit
# Targed to SM36 fixing
#========================================================================================        
    def test13(self):
        import datetime 
        import time

        Work_TZs     = ["UTC", "EST",  "EST5EDT",  "America/Chicago", "America/Los_Angeles"]
        Query_Starts = ["1999-09-16T20:00:00", "2000-01-29T20:00:00"]
        Expect_Nums  = [1, 1]
        Expect_Files = ["acisD1999-09-16p2_respN0005.fits", "acisD2000-01-29p2_respN0006.fits"]
 
        #------
        print("\nTest Start-Time querings in various TimeZone -- ")
        #-------

        # reserve the default TZ value
                 
        tz_default = os.getenv("TZ")
        print("\tCurrent Default TZ :", os.environ.get('TZ', '(not set)'))
        print("\tCurrent Local Time :", time.strftime('%Y-%m-%d %H:%M:%S')) # before timezone change
        print("\n");


        status = 0;
        for index in range(len(Work_TZs)) :
            os.environ["TZ"] = Work_TZs[index]
            time.tzset()

            dst = time.daylight
            print(index+1,") Set timezone: ",  time.tzname, "or '", Work_TZs[index],"'" )
            print('\tSTD-Zone: %d (%d)' % (time.timezone, (time.timezone / 3600)), ',\tDST:', dst)
            print("\n")

            for jamex in range(len(Query_Starts)) :
                #
                # launch caldb4-lib and querying
                #


                c = caldb4.Caldb("chandra", "acis", "sc_matrix") 
                c.CCD_ID=7
                c.start  = Query_Starts[jamex]

                files=c.search
                num = len(files)
                print("   ",jamex+1,"> Querying: StartTime = ",c.start)

                rtn_num   = Expect_Nums[jamex]
                rtn_cfile = Expect_Files[jamex]
                if rtn_num != num: 
                    print ("\tfailed: Expect num files: ", rtn_num)
                    status +=1
                elif num==0 :
                    print ("   failed: 0 found, Expect ",rtn_num," files.")
                    status +=1

                for ii in range(num):
                    if (files[ii].find(rtn_cfile) != -1) :
                        print ("\tRetrieved: ",ii+1," ", os.path.basename(files[ii]))
                    else:
                        print ("\tfailed: Expect: ",ii+1," ",  rtn_cfile); 
                        status +=1

                print("\n");
                c.close
            ## end of for-jamex-loop

            # revert to as was
            if tz_default:
                os.environ["TZ"] = tz_default
            else:
                del os.environ["TZ"] 
        ## end of the for-index-loop

        if status:
            print("NOT-OK")
   

if __name__ == "__main__":

    if "-u" in sys.argv[1:]:
        print (usage() )     
    elif len(sys.argv)==1 or (len(sys.argv)==2 and ("-v" in sys.argv or "-q" in sys.argv ) ):
        unittest.main(defaultTest="TestCaldb4lib")
    else:
        suite = unittest.TestSuite()
        for arg in sys.argv[1:]:
            if arg == "-t1":
                suite.addTest(TestCaldb4lib("test1")) 
                sys.argv.remove("-t1")
            elif arg == "-t2":
                suite.addTest(TestCaldb4lib("test2"))
                sys.argv.remove("-t2")
            elif arg == "-t3":
                suite.addTest(TestCaldb4lib("test3"))
                sys.argv.remove("-t3")
            elif arg == "-t4":
                suite.addTest(TestCaldb4lib("test4"))
                sys.argv.remove("-t4")
            elif arg == "-t5":
                suite.addTest(TestCaldb4lib("test5"))
                sys.argv.remove("-t5")
            elif arg == "-t6":
                suite.addTest(TestCaldb4lib("test6"))
                sys.argv.remove("-t6")
            elif arg == "-t7":
                thepath=os.getenv("TESTIN")
                if thepath != None:
                    suite.addTest(TestCaldb4lib("test7"))
                    sys.argv.remove("-t7")
            elif arg == "-t8":
                suite.addTest(TestCaldb4lib("test8"))
                sys.argv.remove("-t8")
            elif arg == "-t9":
                suite.addTest(TestCaldb4lib("test9"))
                sys.argv.remove("-t9")
            elif arg == "-t10":
                suite.addTest(TestCaldb4lib("test10"))
                sys.argv.remove("-t10")
            elif arg == "-t11":
                suite.addTest(TestCaldb4lib("test11"))
                sys.argv.remove("-t11")
            elif arg == "-t12":
                suite.addTest(TestCaldb4lib("test12"))
                sys.argv.remove("-t12")
            elif arg == "-t13":
                suite.addTest(TestCaldb4lib("test13"))
                sys.argv.remove("-t13")
            else:
                pass

        unittest.TextTestRunner().run(suite)
     
