require( "LuaXML" )
local tables = require('saotrace.suplib.tables')
local create = require('saotrace.raygen.source').create
local os = require('os')

local validate = require('saotrace.raygen.validate')
local va = require('validate.args')

local M = {}


-- simplify the output from LuaXML.
--  * elements become tables
--  * tags become table keywords
--  * element attributes are named entities in the element's table
--  * a sub-element is hoisted into the parent's attributes
--  * elements at the same level with the same tag are turned into lists of elements
--  * if an element has no attributes or sub-elements, just content,
--    the element table is replaced with the content.
local function xml2table( t )

   local r

   if type(t) ~= 'table' then
      r = tonumber(t) or t
   else
      local n = 0
      local multiple = {}
      r = {}
      for k, v in pairs( t ) do

	 if k ~= 0 then

	    n = n + 1

	    local tag = k
	    if type(v) == 'table' then
	       tag = v:tag();
	    end

	    v = xml2table(v)

	    if multiple[tag] ~= nil then
	       if not multiple[tag] then
		  r[tag] = { r[tag] }
		  multiple[tag] = true
	       end
	       table.insert( r[tag], v )
	    else
	       r[tag] = v
	       multiple[tag] = false
	    end
	 end

      end

      if n == 1 and r[1] ~= nil then
	 r = r[1]
      end

   end


   return r
end


function M.load_sources( ... )

   local vobj = validate.vobj:new()
   vobj:setopts{ named = true }

   local vspec = {

      { name = 'srcfile', type = 'string' },


      { name = 'opts',
	optional = true,

	vtable = {

	   extent = { multiple = true,
		      type = 'function',
		      default = require('saotrace.raygen.extent')
		   },

	   spectrum = { multiple = true,
			type = 'function',
			default = require('saotrace.raygen.spectrum')
		     },

	   position = { type = 'table',
			optional = true,
		     },

	   before = { type = 'function',
		      optional = true },
	}

     },

   }
   local ok, args = vobj:validate( vspec, ... )

   if not ok then
      error( args )
   end

   local sources

   if args.srcfile:find( '%.xml$') then

      local stuff = assert( xml.load( args.srcfile ) )

      sources = assert( stuff:find( 'sources' ), "no <sources> tag in " .. args.srcfile )

   else

      error( "don't know how to load sources from " .. args.srcfile )

   end

   if nil == args.opts.position then
      args.opts.position = {
	 vtable = {
	    -- accept everything else; don't want to
	    -- validate position parameters here
	    ['%named'] = function()
			    return true, {}
			 end
	 }
      }

   end

   for _, source in ipairs( sources ) do

      if source:tag() ~= 'source' then
	 error( "<" .. source:tag() .. "> is not a source element in:\n" .. tostring( source ) )
      end

      local srcpars = xml2table( source )

      -- spectrum must be turned into a list of spectra.  if there
      -- are multiple spectrum entries in the source file xml2table
      -- will already have done this (which is indicated if the
      -- returned spectrum element has element 1 not nil). otherwise
      -- have to do it here
      if srcpars.spectrum ~= nil and srcpars.spectrum[1] == nil then
	 srcpars.spectrum = { srcpars.spectrum }
      end

      -- backwards compatibility mode
      if srcpars.type ~= nil then
	 if srcpars.type ~= "point" then
	    error( "unknown srcpars type: " .. srcpars.type )
	 else
	    srcpars.extent = { type = srcpars.type }
	    srcpars.type = nil
	 end
      end

      -- see if the caller wants to modify any source parameters
      if args.opts.before then
	 srcpars = args.opts.before( srcpars, source )
      end

      local vspec = {

	 name = { type = { 'string', 'number' },
		  after = function( arg ) return true, tostring( arg ) end,
	       },

	 position = function( v )
		       if "table" == type(v) and v[1] ~= nil then
			  return false, "only one position may be specified"
		       end
		       return true, args.opts.position
		    end,

	 spectrum = {
	    multiple = true,
	    min = 1,
	    vtable = {
	       type = { enum = tables.keys( args.opts.spectrum.vspec ) },

	       -- accept everything else; don't want to
	       -- validate spectrum parameters here
	       ['%named'] = function(args)
			       return true, {}
			    end
	    }
	 },

	 extent   = {
	    vtable = {

	       -- if there's a positional element that means there's more than
	       -- one extent, which is bogus
	       ['%pos'] = function()
			     return false, "only one extent may be specified"
			  end,

	       -- just validate the 'type' parameter
	       ['%named'] = function(k, v, vfargs)
			       if k == 'type' then
				  return true, { enum = tables.keys( args.opts.extent.vspec ) }
			       else
				  return true, {}
			       end
			    end
	    },
	 },
      }

      local ok, srcpars = vobj:validate_tbl( vspec, srcpars )
      if not ok then
	 error( "error in source parameters: "..srcpars.." in:\n" .. tostring(source) )
      end


      local extent_type = srcpars.extent.type
      srcpars.extent.type = nil

      -- if there's no extent name, use that of the source
      srcpars.extent.name = srcpars.extent.name or srcpars.name
      srcpars.extent.position = srcpars.position

      local ok, extent = pcall( args.opts.extent[extent_type], srcpars.extent )
      if not ok then
	 error( "errors in source parameters: "..extent.. "in: \n" .. tostring(source) )
      end

      local src_spectra = {}
      for idx, spec in ipairs( srcpars.spectrum ) do

	 local spec_type = spec.type
	 spec.type = nil

	 -- if there's no spectrum name, cobble one together
	 spec.name = spec.name or srcpars.name .. tostring( idx )

	 local ok, spectrum = pcall( args.opts.spectrum[spec_type], spec )

	 if not ok then
	    error( spectrum .. " in:\n" .. tostring( source ) )
	 end

	 table.insert( src_spectra, spectrum )

      end

      local ok, msg = pcall( create, srcpars.name, { extent = extent,
						     spectrum = src_spectra } )

      if not ok then
	 error( msg .. " in:\n" .. tostring( source ) )
      end

   end

end

if os.getenv( 'TESTING' ) then
   print ("TESTING" )
   M.xml2table = xml2table
end

return M
