local math     = require('math')
local string   = require('string')
local ut       = require('udunits2')
local units    = require('saotrace.raygen.units')

-- setup for unit conversions using udunits2

-- create our own specialized validation types
local va = require( 'validate.args' )
local vobj = va:new()

vobj:setopts{ named = true }


vobj:add_type( 'eap_id',
	     function( arg )
		return va.zposnum( arg ) and arg >= 1000, arg
	     end
	  )


local function posquant( arg )
   if arg[1] > 0 then
      return true, arg
   else
      return false, "not a positive number"
   end
 end

local function zposquant( arg )

   if arg[1] >= 0 then
      return true, arg
   else
      return false, "not a non-negative number"
   end
 end


-- angular
for k,v in pairs( units.angular ) do

   -- add plain angle, no constraints
   vobj:add_type( 'angle_' .. k,
		  function (arg)
		     return units.convert_angle( arg, v, units.angular.radian )
		  end
	       )

   -- add positive angle
   vobj:add_type( 'pos_angle_' .. k,
		  function (arg)
		     return units.convert_angle( arg, v, units.angular.radian, posquant )
		  end
	       )

   -- add non-negative angle
   vobj:add_type( 'zpos_angle_' .. k,
		  function (arg)
		     return units.convert_angle( arg, v, units.angular.radian, zposquant )
		  end
	       )
end

local function validate_angle_ra( arg )

   -- first see if it's in H:M:S format
   if type(arg) == 'string' then
      local h, m, s = arg:match( '^(%d+):(%d+):([%d.]+)$' )
      if h ~= nil then
	 h, m, s = tonumber(h), tonumber(m), tonumber(s)
	 if s == nil then
	    return false, "can't parse angle: " .. arg
	 end
	 arg = (h + ( m  + s / 60 ) / 60 ) * 15
      end
   end

   return units.convert_angle( arg, units.angular.degree, units.angular.radian, zposquant )

end

vobj:add_type( 'angle_ra', validate_angle_ra )


local function validate_angle_dec( arg )
   local valid_dec = function( dec )
			if dec[1] < -math.pi/2 or dec[1] > math.pi/2 then
			   return false, "declination outside of [-pi/2,pi/2]"
			else
			   return true, dec
			end
		     end


   -- first see if it's in H:M:S format
   if type(arg) == 'string' then
      local sign, h, m, s = arg:match( '^([+-]?)(%d+):(%d+):([%d.]+)$' )
      if h ~= nil then
	 h, m, s = tonumber(h), tonumber(m), tonumber(s)
	 if s == nil then
	    return false, "can't parse angle: " .. arg
	 end
	 sign = sign == '-' and -1 or 1
	 arg = sign * (h + ( m  + s / 60 ) / 60 )
      end
   end

   return units.convert_angle( arg, units.angular.degree, units.angular.radian, valid_dec )
end

vobj:add_type( 'angle_dec', validate_angle_dec )


-- time metric
vobj:add_type( 'time',
	     function (arg)
		-- capture any errors; message will be in variable value if so
		local ok, value = pcall( units.convert, arg, units.time.seconds, units.time.seconds, zposquant )
		return ok, value
	     end
	  )


-- linear distance metric
vobj:add_type( 'distance',
	     function (arg)
		-- capture any errors; message will be in variable value if so
		local ok, value = pcall( units.convert, arg, units.linear.mm, units.linear.mm )
		return ok, value
	     end
	  )

-- angular or linear distance metric
vobj:add_type( 'ang_lin_distance',
	     function (arg)
		local ok, value, input_unit = pcall( units.normalize_value, arg, nil )
		if not ok then
		   return false, value
		end

		if input_unit == nil then
		   return false, "no unit specified"
		end

		-- make sure it's an angle or a distance
		ok =    input_unit:are_convertible( units.angular.radian )
		     or input_unit:are_convertible( units.angular.mm )

		if not ok then
		   return false, "input unit is not an angular or linear distance unit"
		end

		return true, { value, input_unit }
	  end
	  )


vobj:add_type( 'axis',
	     function (arg )

		if ( type(arg) ~= 'table' ) then
		   return false, "not a list of axis coords"
		end

		for k,v in pairs( arg ) do

		   if type(k) ~= 'number' or ( k ~= 1 and k ~= 2 and k ~= 3 ) then
		      return false, "unexpected key in list: " .. k .. " = " .. v
		   end

		   if type(v) ~= 'number' or ( math.abs(v) ~= 1 and v ~= 0 ) then
		      return false,
		        string.format("axis coordinate value (%s) is not one of -1, 0, or 1",
				      v)
		   end

		end
		return true, arg
	     end
	  )

vobj:add_type( 'axisangle',
	       function ( arg )

		  if ( type(arg) ~= 'table' ) then
		     return false, "not an axis angle"
		  end

		  -- create a new object; don't interrupt processing of current validation

		  local nvobj = vobj:new()

		  return nvobj:validate_tbl( {
						angle = { type = 'angle_radian' },
						axis  = { type = 'axis' },
					     }, arg )
	       end
	    )

local idist_validate = require( 'saotrace.raygen.validate.idist').validate

vobj:add_type( 'idist',
	     function( ... )
		return idist_validate( vobj, ... )
	     end )

return {

   vobj = vobj,
   posquant = posquant,
   zposquant = zposquant,
   validate_angle_ra = validate_angle_ra,
   validate_angle_dec = validate_angle_dec,

}
