package RDB;

# ABSTRACT: object methods for dealing with rdb files

use Safe::Isa;
use Scalar::Util;

use fields qw( fh col defs ncols pos comments loc mode file bindvals
  bindmap bindsub nsplitcols hdrvars hdr_written );

use strict;
use warnings;

our $VERSION = '1.53';

my $DEFN_RE
  = '^\s*'
  . '(?:(\d+)\s+)?'
  . '([\w%:@=,.][\w%:@=,.#-]*)' . '\s+' . '(\d*)'
  . '([SsNnMm])'
  . '([<=>]?)'
  . '(?:\s*$|\s+(.+))'
  . '$';    #'


use IO::File ();
use Carp     ();



#pod =method new
#pod
#pod   $rdb = RDB->new( [$data, [$mode] | \@defs ] );
#pod
#pod Create an B<RDB> object, optionally attaching it to a data source.
#pod If C<$data> (and optionally C<$mode> is specified, the B<open> method is invoked.
#pod If C<@defs> is passed, the B<init> method is invoked.
#pod
#pod C<$data> may be one of the following:
#pod
#pod =over
#pod
#pod =item  I<filename>
#pod
#pod The name of a file (a scalar)
#pod
#pod =item I<scalar ref>
#pod
#pod A string containing RDB formatted data.  Pass a I<reference> to the string.
#pod
#pod =item I<file handle>
#pod
#pod An existing, opened file.
#pod
#pod =back
#pod
#pod C<$mode> defaults to C<< < >>.
#pod
#pod =cut

sub new {
    my $this  = shift;
    my $class = ref( $this ) || $this;
    my $rdb   = {
        fh          => undef,
        col         => [],
        defs        => {},
        ncols       => undef,
        'pos'       => {},
        comments    => [],
        hdrvars     => {},
        loc         => undef,
        mode        => undef,
        file        => undef,
        bindmap     => [],
        bindsub     => undef,
        attr        => { AlwaysBind => 0 },
        hdr_written => 0,                  # true if header has been written
        nsplitcols  => 0,                  # number of columns to pass to split;
                                           # used only if variables are bound to
                                           # columns
    };

    bless $rdb, $class;

    my $arg = shift;

    if ( defined $arg ) {

        # if it's an existing RDB object, copy its innards
        if ( $arg->$_isa( 'RDB' ) ) {
            $rdb->init( $arg );
        }
        else {
            my $mode = shift;
            $rdb->open( $arg, $mode ) or return;
        }

    }

    return $rdb;
}


#pod =method bind
#pod
#pod    $rdb->bind( \%bindhash [, \%attrs ] )
#pod
#pod B<bind> simplifies the processing of rdb files by allowing the
#pod automatic assignment of values read from the rdb file to Perl
#pod variables or arrays. Each time that the read method is called with no
#pod arguments, it will update the variables specified in preceding calls
#pod to C<bind>.  B<bind> takes a hash of columns to be bound; the keys are
#pod the column names, their values are references to either scalars or
#pod arrays.  In the former case, the scalar will be assigned the column's
#pod value.  In the latter case, the column's value is pushed onto the end
#pod of the array.  ( Note that the argument to bind is a hash just to
#pod enforce the correct number of items.)  For example,
#pod
#pod         $rdb->bind( { col1 => \$col1, col2 => \$col2 } );
#pod         while ( $rdb->read( ) )
#pod         {
#pod           print "$col1, $cols\n";
#pod         }
#pod
#pod Or, using arrays,
#pod
#pod         my ( @col1, @col2 );
#pod         $rdb->bind( { col1 => \@col1, col2 => \@col2 } );
#pod         1 while ( $rdb->read( ) );
#pod         for( $i = 0 ; $i < @col1 ; $i++ )
#pod         {
#pod           print $col1[$i], ' ', $col2[$i], "\n";
#pod         }
#pod
#pod
#pod If the same column is specified in I<succeeding> calls to B<bind>, the
#pod new binding will override the previous binding.
#pod
#pod However, if the same column should be bound to multiple variables, the
#pod C<Override> attribute may be reset using the second argument to
#pod B<bind>:
#pod
#pod         $rdb->bind( { col1 => \$col1, col2 => \$col2 } );
#pod         $rdb->bind( { col1 => \$col1_copy }, { Override => 0 } );
#pod
#pod The column C<col1> will now be written to both C<$col1> and C<$col1_copy>.
#pod
#pod =cut

sub bind {
    @_ == 2 || @_ == 3
      or Carp::croak 'usage: $rdb->bind( \%bindmap [, \%attrs] )';

    my $rdb     = shift;
    my $bindmap = shift;

    Carp::croak 'RDB::bind: argument not a hash'
      unless ref $bindmap eq 'HASH';

    my $attrs = shift;
    Carp::croak 'RDB::bind: attrs not a hash'
      if $attrs && ref $attrs ne 'HASH';

    $attrs = { Override => 1, $attrs ? %$attrs : () };

    my @bindmap = @{ $rdb->{bindmap} };

    # if overriding existing map, retain entries from old map which
    # aren't being overridden
    if ( $attrs->{Override} ) {
        my @new;
        while ( @bindmap ) {
            my $bindval = shift @bindmap;
            push @new, $bindval
              unless $bindmap->{ $bindval->[0] };
        }
        @bindmap = @new;
    }


    while ( my ( $col, $var ) = each %$bindmap ) {
        push @bindmap, [ $col, $var ];
    }

    $rdb->{bindmap} = \@bindmap;

    $rdb->__bind_sub;
}


# optimizations here must take into account a non one-to-one mapping
# i.e., more than one variable may be bound to a column.
sub __bind_sub {
    my $rdb = shift;

    my @sub;

    return unless @{ $rdb->{bindmap} };

    for ( my $i = 0 ; $i < @{ $rdb->{bindmap} } ; $i++ ) {
        my ( $col, $var ) = @{ $rdb->{bindmap}[$i] };

        Carp::croak( "RDB::bind: column `$col' not defined in rdb file" )
          unless exists $rdb->{'pos'}->{$col};

        $rdb->{nsplitcols} = $rdb->{'pos'}{$col} + 2
          if $rdb->{nsplitcols} < $rdb->{'pos'}{$col} + 2;

        if ( 'SCALAR' eq ref( $var ) ) {
            push @sub,
              sprintf( '${$rdb->{bindmap}[%d][1]} = $data[%d];',
                $i, $rdb->{'pos'}->{$col} );
        }
        elsif ( 'ARRAY' eq ref( $var ) ) {
            push @sub,
              sprintf( 'push @{$rdb->{bindmap}[%d][1]}, $data[%d];',
                $i, $rdb->{'pos'}->{$col} );
        }
        else {
            Carp::croak(
                "RDB::bind: column `$col' -> must bind to \\\@ or \\\$" );
        }
    }

    my $sub = join( "\n",
        'use integer;',
        'my $rdb = shift;',
        qq{my \@data = split( "\\t", shift, $rdb->{nsplitcols});}, @sub );

    $rdb->{bindsub} = $rdb->__make_sub( $sub );
    Carp::croak if $@;
}

sub __make_sub {
    my ( $rdb, $statements ) = @_;

    my $sub;

    ## no critic ( ProhibitStringyEval )
    eval qq( \$sub = sub { $statements } );
    return $sub;
}

#pod =method close
#pod
#pod explicitly close an rdb file.  This usually need not be called, as the
#pod file will be closed when the RDB object is destroyed.
#pod
#pod =cut

sub close {
    1 == @_ or Carp::croak 'usage: $rdb->close()';

    my ( $rdb ) = @_;

    if ( $rdb->{fh} ) {
        $rdb->write_hdr if _is_write( $rdb->{mode} ) && !$rdb->{hdr_written};
        $rdb->{loc} = tell( $rdb->{fh} );
        close $rdb->{fh};
        $rdb->{fh} = undef;
    }

    return;
}

sub _is_write {
    my $mode = shift;
    return $mode =~ />/;
}

#pod =method init
#pod
#pod   $rdb-> init( @defs | \@defs | $rdb );
#pod
#pod Initialize the rdb object with a set of columns.  A column is
#pod specified by both a name and a definition.  Definitions technically
#pod consist of four parts: the column name, it's type, output alignment,
#pod and description.  The latter are optional and are usually omitted.
#pod Column types are one of C<N>, C<S>, or C<M>, for numeric, string, and
#pod month data.  Alignment is one of C<<> or C<>>.
#pod
#pod B<init> is passed either an array (or list), an array reference, or a
#pod reference to another B<RDB> object.  In the latter case, the column
#pod definitions of the other object are duplicated.  In the former cases,
#pod the array must contain column name and definition pairs.
#pod
#pod The definition may take any of the following forms:
#pod
#pod =over 8
#pod
#pod =item *
#pod
#pod If the definition is a scalar, it should be the column type:
#pod
#pod   $rdb->init( c1 => 'N' );
#pod
#pod =item *
#pod
#pod if the definition is a hash reference, it should have at least
#pod the key C<type>, and may optionally have the keys C<width>, C<align>,
#pod or C<desc>
#pod
#pod   $rdb->init( c1 => { type => 'N', width => 3, align => '<',
#pod                       desc => 'This column is meaningless } );
#pod
#pod =item *
#pod
#pod if the definition is an array reference, it may have up to four
#pod elements; the type, width, alignment, and description, in order.
#pod The last three are optional.
#pod
#pod   $rdb->init( c1 => [ 'N', 3, '<', 'This column is meaningless' ] );
#pod
#pod =back
#pod
#pod Any of these forms may be mixed:
#pod
#pod   $rdb->init( c1 => 'N',
#pod               c2 => [ 'N', 32 ],
#pod               c3 => { type => 'N', desc => 'What A Nice Column' } );
#pod
#pod =cut

sub init {
    2 <= @_ or Carp::croak 'usage: $rdb->init(@defs|\@defs)';
    my $rdb = shift;

    $rdb->{ncols}    = 0;
    $rdb->{col}      = [];
    $rdb->{defs}     = {};
    $rdb->{pos}      = {};
    $rdb->{comments} = [];
    $rdb->{hdrvars}  = {};

    if ( $_[0]->$_isa( 'RDB' ) ) {
        my $orig = $_[0];
        $rdb->{comments} = [ @{ $orig->{comments} } ];
        $rdb->{hdrvars}  = { %{ $orig->{hdrvars} } };
    }


    eval { $rdb->_addcols( @_ ); };
    Carp::croak "RDB::init $@" if $@;
}

#pod =method init_tpl
#pod
#pod   $rdb->init_tpl( $file_name | \$tpl_string )
#pod
#pod Initialize an RDB object from an RDB header template. If the passed
#pod argument is a scalar, it should contain the name of a file containing
#pod the template.  If it's a reference it should be a reference to a
#pod scalar containing the template.  An RDB header template is description
#pod of the header in the following format.
#pod
#pod Each column is specified on a separate line, and contains up to
#pod four white space delimited fields:
#pod
#pod =over 8
#pod
#pod =item 1
#pod
#pod an optional field containing the column's zero based index.  If not specified,
#pod the ordering of the field in the template is used.  For example,
#pod
#pod    fee S
#pod    fie N
#pod    fo  N
#pod    fum N
#pod
#pod is equivalent to
#pod
#pod    0 fee S
#pod    1 fie N
#pod    2 fo  N
#pod    3 fum N
#pod
#pod Be careful when mixing lines with and without an index:
#pod
#pod      fee S
#pod    2 fie N
#pod      fo  N
#pod      fum N
#pod
#pod is equivalent to
#pod
#pod    0 fee S
#pod    2 fie N
#pod    2 fo  N
#pod    3 fum N
#pod
#pod which will result in an error.  Indices must be unique.
#pod
#pod There's a further degeneracy which must be avoided:
#pod
#pod    3 N S
#pod
#pod Is that an index of C<3>, a name of C<N> and a type of C<S>, or is
#pod that a name of C<3>, a type of C<N> and a description of C<S>?
#pod It is parsed as the former.  To get the latter interpretation,
#pod you'll have to include an index field.
#pod
#pod
#pod =item 2
#pod
#pod the column name.  it may appear in quotes.
#pod
#pod =item 3
#pod
#pod the column type.  it may include the column width as a prefix
#pod
#pod =item 4
#pod
#pod an optional column description
#pod
#pod =back
#pod
#pod Comment lines may be present, and are indicated by a leading C<#> character.
#pod
#pod For example,
#pod
#pod   # P-to-H Decenter parameters derived from XRCF HSI off-axis images
#pod   # (single shell); used pitch=0, yaw=-20 arcmin data.
#pod   #
#pod    0               fee  6S      what i get paid
#pod    1               fie 10N      upon you
#pod    2               fo  10N      fight or no?
#pod    3               fum  9N      ble
#pod
#pod =cut


sub init_tpl {
    my $rdb = shift;

    my $tpl = shift;

    $rdb->{ncols}    = 0;
    $rdb->{col}      = [];
    $rdb->{defs}     = {};
    $rdb->{pos}      = {};
    $rdb->{comments} = [];
    $rdb->{hdrvars}  = {};

    my @cols;

    local $_;

    my $idx = 0;
    unless ( 'SCALAR' eq ref $tpl ) {
        my $fh = new IO::File $tpl, "r"
          or Carp::croak( "unable to open $tpl\n" );

        while ( <$fh> ) {
            chomp;
            $rdb->_parse_template( \@cols, $idx++, $_ );
        }
    }
    else {
        $rdb->_parse_template( \@cols, $idx++, $_ ) for split( "\n", $$tpl );
    }

    # ensure that there are no duplicate indices
    my %idxs = map { ( $_->{index}, 1 ) } @cols;
    Carp::croak( "duplicate index values in template\n" )
      if keys %idxs != @cols;

    $rdb->_addcols(
        map { ( $_->{name}, $_ ) }
        sort { $a->{index} <=> $b->{index} } @cols
    );
}

sub _parse_template {
    my ( $rdb, $cols, $idx, $tpl ) = @_;

    local $_ = $tpl;

    # store comments
    if ( /^\s*\#/ ) {
        s/^#//;
        $rdb->add_comments( $_ );
    }
    else {

        if ( $tpl =~ /$DEFN_RE/ ) {
            push @$cols,
              {
                index => defined $1 ? $1 : $idx,
                name  => $2,
                width => $3,
                type  => $4,
                align => $5,
                desc  => $6
              };
        }
        else {
            Carp::croak( "illegal template definition `$tpl'\n" );
        }

    }
}


#pod =method write_tpl
#pod
#pod   $rdb->write_tpl( $filename | $fh);
#pod
#pod Write an RDB template for the current RDB object.  The argument may be
#pod a scalar, it which case it should contain the name of a file to which
#pod to write the template, or a filehandle.
#pod
#pod =cut

sub write_tpl {
    my $rdb = shift;

    local $_;

    my $fh = shift;
    unless ( ref $fh ) {
        my $fh_t = new IO::File $fh, "w"
          or Carp::croak( "unable to create $fh\n" );
        $fh = $fh_t;
    }
    $rdb->_write_hdr_comments( $fh );

    # turn off warnings here; don't care about undefined values, as
    # they are just attributes that haven't been defined.
    local $^W = 0;

    # index, name, type, desc
    my %max = ( name => 0, width => 0 );
    foreach ( @{ $rdb->{col} } ) {
        $max{name} = length( $_ ) if $max{name} < length( $_ );
        $max{width} = length( $rdb->{defs}{$_}{width} )
          if $max{width} < length( $rdb->{defs}{$_}{width} );
    }

    foreach ( sort { $rdb->{pos}{$a} <=> $rdb->{pos}{$b} } @{ $rdb->{col} } ) {
        printf $fh "%4d\t%$max{name}s\t%-$max{width}s%s%s\t%s\n",
          $rdb->{pos}{$_}, $_,
          $rdb->{defs}{$_}{width},
          $rdb->{defs}{$_}{type},
          ( $rdb->{defs}{$_}{align} || ' ' ),
          $rdb->{defs}{$_}{desc};
    }

    return;
}

#pod =method add_col
#pod
#pod   $rdb->add_col( @defs | \@defs | $rdb);
#pod
#pod Add new columns to the rdb object.  See the description of the
#pod B<init()> method for the specification of the column names and
#pod definitions.  Existing columns are not duplicated; their definitions
#pod are changed to the passed type.
#pod
#pod =cut

sub add_col {
    2 <= @_ or Carp::croak 'usage: $rdb->add_col(@defs|\@defs)';
    my $rdb = shift;

    eval { $rdb->_addcols( @_ ); };
    Carp::croak "RDB::add_col $@" if $@;
}

sub _addcols {
    my $rdb = shift;

    local $_;

    my @defs;

    if ( 2 <= @_ ) {
        @defs = @_;
        @defs % 2 and die 'missing definition';
    }
    else {
        ref( $_[0] ) or die 'argument must be list or reference to array';
        if ( ref $_[0] eq 'ARRAY' ) {
            @defs = @{ $_[0] };
        }
        else {
            my $src = shift;

            if ( UNIVERSAL::isa( $src, 'RDB' ) ) {
                push @defs, map { $_, $src->{defs}{$_} } @{ $src->{col} };
            }
            else {
                die( "don't know what to make of argument" );
            }
        }
    }

    my @fields = qw( type width align desc );

    # slurp column name, definition pairs
    while ( my ( $col, $def ) = splice( @defs, 0, 2 ) ) {
        Carp::croak( "RDB::init: undefined column name\n" )
          unless defined $col;

        Carp::croak( "RDB::init: undefined column definition\n" )
          unless defined $def;

        unless ( exists $rdb->{defs}{$col} ) {
            push @{ $rdb->{col} }, $col;
            $rdb->{pos}{$col} = $rdb->{ncols};
            $rdb->{ncols}++;
        }

        # it's a hash; it should have 'width', 'type', 'align', 'desc' fields
        if ( 'HASH' eq ref $def ) {
            $rdb->{defs}{$col}{$_} = $def->{$_} || '' foreach @fields;
        }

        # it's an array; it should have 4 fields, as above
        elsif ( 'ARRAY' eq ref $def ) {
            my @tmp = @{$def};
            $rdb->{defs}{$col}{$_} = shift @tmp || '' foreach @fields;
        }

        # any other type of ref is bogus
        elsif ( ref $def ) {
            Carp::croak( "RDB::init: illegal reference in def for `$col'" );
        }

        # ordinary scalar, it's the type
        else {
            @{ $rdb->{defs}{$col} }{@fields} = ( '', $def, '', '' );
        }
    }

    return;
}



#pod =method delete_col
#pod
#pod   $rdb-> delete_col( @cols );
#pod
#pod delete the specified columns from the object.  This is only applicable
#pod to RDB files open for writing, and only before the RDB header has
#pod been written out.
#pod
#pod         $rdb->delete_col( 'a', 'b' );
#pod
#pod =cut

sub delete_col {
    @_ > 1 or Carp::croak 'RDB::delete_col( @cols )';

    local $_;

    my $rdb  = shift;
    my @cols = @_;

    return
      if ( $rdb->{mode} && !_is_write( $rdb->{mode} ) )
      || $rdb->{hdr_written};

    foreach my $col ( @cols ) {
        Carp::croak( "RDB::delete_col `$col' not defined" )
          unless exists $rdb->{'pos'}->{$col};

        delete $rdb->{'defs'}->{$col};

        $rdb->{bindmap} = [ grep { $_->[0] ne $col } @{ $rdb->{bindmap} } ];
        $rdb->{col}     = [ grep { $_ ne $col } @{ $rdb->{col} } ];
        $rdb->__bind_sub;
        $rdb->{ncols}--;
    }
    $rdb->{'pos'} = {};
    @{ $rdb->{'pos'} }{ @{ $rdb->{'col'} } } = ( 0 .. $#{ $rdb->{col} } );

    return;

}

#pod =method set
#pod
#pod   $rdb-> set( \%attr );
#pod
#pod B<set> specifies the values of various attributes for the object.
#pod The passed reference should point to a hash which may contain
#pod the following keys:
#pod
#pod =over
#pod
#pod =item AlwaysBind
#pod
#pod If this is set, e.g.,
#pod
#pod   $rdb->set( { AlwaysBind => 1} )
#pod
#pod if B<RDB::bind> has been called to set up bindings between columns and
#pod Perl variables, the Perl variables will always be updated, regardless of which
#pod form of B<RDB::read> is called.
#pod
#pod =back
#pod
#pod =cut

sub set {
    my $rdb = shift;

    my $attr = shift;

    Carp::croak 'usage $rdb->attr( \%attr )' unless 'HASH' eq ref $attr;

    $rdb->{attr} = { %{ $rdb->{attr} }, $attr ? %$attr : () };

    return;
}


#pod =method open
#pod
#pod   $rdb->open( $data, [ $mode ] );
#pod
#pod Attach the object to a data source.
#pod
#pod C<$data> may be one of the following:
#pod
#pod =over
#pod
#pod =item  I<filename>
#pod
#pod The name of a file (a scalar)
#pod
#pod =item I<scalar ref>
#pod
#pod A string containing RDB formatted data.  Pass a I<reference> to the string.
#pod
#pod =item I<file handle>
#pod
#pod An existing, opened file.
#pod
#pod =back
#pod
#pod Modes are the standard Perl-ish ones (see the Perl open command).  If
#pod the mode is read only or read/write, it reads and parses the RDB
#pod header.  C<$mode> defaults to C<< < >>.
#pod
#pod It returns C<undef> upon error.
#pod
#pod =cut

sub open {
    @_ >= 2 or Carp::croak 'usage : $rdb->open( $file [, $mode] )';
    my ( $rdb, $file, $mode ) = @_;

    local $_;

    $mode = '<' if !defined $mode;
    $rdb->{mode} = $mode;

    my $fh;

    # want the underling IO object for a GLOB
    $file = *{$file}{IO}
      if 'GLOB' eq ref $file;

    # scalar => file name
    if ( '' eq ref $file ) {

        $fh = IO::File->new( $file, $mode );

    }

    # reference to scalar => a string containing the RDB table
    elsif ( 'SCALAR' eq ref( $file ) ) {

        require IO::String;
        $fh = IO::String->new( $$file );

    }

    # if it has a fileno method, assume it's something we can deal with
    elsif ( $file->$_can( 'fileno' ) ) {

        $fh = IO::File->new->fdopen( $file->fileno, $mode );
        $fh->seek( $file->tell, 0 );
    }

    else {

        Carp::croak( "don't know how to handle passed $file thing\n" );

    }

    return if !defined $fh;


    $rdb->{file} = $file;

    # if this is open for reading, suck in the header
    if ( $mode =~ /[<+]/ ) {
        $rdb->{comments} = [];
        while ( <$fh> ) {
            last unless /^\s*\#/;
            chop;

            s/^\s*#//;
            push @{ $rdb->{comments} }, $_;

            # grab header variable
            if ( /^:\s*(\w+)\s*=\s*(.*)/ ) {
                $rdb->{hdrvars}{$1} = $2;
            }
        }
        return unless defined $_;
        $rdb->{'col'}  = [];
        $rdb->{'defs'} = {};
        $rdb->{'pos'}  = {};

        chop;
        @{ $rdb->{'col'} } = split( "\t" );
        @{ $rdb->{'pos'} }{ @{ $rdb->{'col'} } } = ( 0 .. $#{ $rdb->{col} } );
        $_ = <$fh>;
        return unless defined $_;
        chop;

        # chop up definition into width, type, alignment, and description
        @{ $rdb->{'defs'} }{ @{ $rdb->{'col'} } } = map {
            /(\d+)?([nNsSmM]|-+)([<>])?(?:\s+(.*))?/;
            {
                width => $1,
                type  => $2,
                align => $3,
                desc  => $4
            }
        } split( "\t" );

        # ensure that we got valid definitions
        while ( my ( $col, $def ) = each( %{ $rdb->{defs} } ) ) {
            Carp::croak "unrecognized definition for column `$col'"
              unless defined $def->{type};
            # handle old /rdb format
            $def->{type} = 'S' if $def->{type} =~ /-+/;
        }

        $rdb->{start} = tell;
        $rdb->{ncols} = @{ $rdb->{'col'} };
    }

    $rdb->{fh}  = $fh;
    $rdb->{loc} = tell( $rdb->{fh} );

    return 1;
}

#pod =method read
#pod
#pod   $rdb->read( [ %data | @data ] );
#pod
#pod Read in the next line from the rdb database, storing the columns into
#pod either a hash keyed off of the column names (if passed a reference to
#pod the hash), an array (if passed a reference to the array), or into
#pod scalars specified by previous calls to the C<bind> method (if C<read>
#pod is called with no arguments). It returns the undefined value upon end
#pod of file.  It does not check to ensure that there are enough columns in
#pod the input.  For example:
#pod
#pod         $rdb->read(\%data);
#pod         print "foo = $data{foo}\n";
#pod
#pod         $rdb->read(\@data);
#pod         print "The first column has value $data[0]\n";
#pod
#pod         $rdb->bind( { foo => \$foo } );
#pod         $rdb->read();
#pod         print "Foo = $foo\n";
#pod
#pod =cut

sub read {
    @_ >= 1 or Carp::croak 'usage : $rdb->read( [\%data|\@data] )';

    local $_;

    my $rdb = shift;

    my $fh = $rdb->{fh};
    return unless defined( $_ = <$fh> );
    chomp;
    $rdb->{loc} = tell( $rdb->{fh} );

    my $data = shift;

    if ( ref $data eq 'HASH' ) {
        @{$data}{ @{ $rdb->{col} } } = split( "\t", $_, $rdb->{ncols} );
    }
    elsif ( ref $data eq 'ARRAY' ) {
        @{$data} = split( "\t", $_, $rdb->{ncols} );
    }

    if ( !defined $data || $rdb->{attr}{AlwaysBind} ) {
        # use bound variables to return data

        Carp::croak 'RDB::read no variables have been bound'
          unless defined $rdb->{bindsub};

        &{ $rdb->{bindsub} }( $rdb, $_ );
    }

    return 1;
}

#pod =method read_line
#pod
#pod   $rdb->read_line;
#pod
#pod C<read_line> reads a line from the rdb file without parsing it (even
#pod to chop off the end).  It returns C<undef> upon end of file.
#pod
#pod =cut

sub read_line {
    @_ == 1 or Carp::croak 'usage : $rdb->read_line(  )';

    local $_;

    my $rdb = shift;

    my $fh = $rdb->{fh};
    $_ = <$fh>;
    $rdb->{loc} = tell( $rdb->{fh} );

    return unless defined $_;
    chomp;

    return $_;
}

#pod =method rename
#pod
#pod   $rdb->rename( \%renamehash );
#pod
#pod B<rename> is passed a hash of columns to be renamed; the keys are
#pod the old column names, their values are the new names.  It's a hash
#pod just to enforce the correct number of items.  For example,
#pod
#pod         $rdb->rename( { oldcol => 'newcol', foocol => 'boocol' } );
#pod
#pod =cut

sub rename {
    @_ == 2 or Carp::croak 'RDB::rename( \%renamevals )';

    local $_;

    my $rdb     = shift;
    my $renames = shift;

    Carp::croak 'RDB::rename: argument not a hash'
      unless ref $renames eq 'HASH';

    while ( my ( $old, $new ) = each %$renames ) {
        # ignore 'em if they're the same; saves grief later
        next if $old eq $new;

        Carp::croak( "RDB::rename `$old' not defined in rdb file" )
          unless exists $rdb->{'pos'}->{$old};

        Carp::croak( "RDB::rename '$new' already defined in rdb file" )
          if exists $rdb->{'pos'}->{$new};

        $rdb->{'defs'}->{$new} = $rdb->{'defs'}->{$old};
        delete $rdb->{'defs'}->{$old};

        $rdb->{'pos'}->{$new} = $rdb->{'pos'}->{$old};
        delete $rdb->{'pos'}->{$old};

        foreach ( @{ $rdb->{bindmap} } ) {
            $_->[0] = $new if $_->[0] = $old;
        }
    }

    foreach ( @{ $rdb->{'col'} } ) {
        $_ = $renames->{$_} if exists $renames->{$_};
    }
}


#pod =method reopen
#pod
#pod   $rdb-> reopen;
#pod
#pod B<reopen> reopens a file that has previously been opened and closed,
#pod positioning the filepointer to where it was before it was closed.  It
#pod retains the previous access mode.  It does not reopen files passed to
#pod the original call of rdb::open as references.  It returns the
#pod undefined value upon error.
#pod
#pod =cut

sub reopen {
    @_ == 1 or Carp::croak 'usage : $rdb->reopen()';
    my ( $rdb ) = @_;

    return unless defined $rdb->{loc};

    my $file = $rdb->{file};

    if ( '' eq ref $file ) {

        $rdb->{fh} = IO::File->new( $file, $rdb->{mode} );

    }

    elsif ( 'SCALAR' eq ref( $file ) ) {

        $rdb->{fh} = IO::String->new( $$file );

    }

    elsif ( Scalar::Util::blessed $file ) {

        Carp::croak( "can't reopen a passed object once it has been closed\n" );

    }

    return if !defined $rdb->{fh};


    # if $rdb->{loc} is -1, at EOF (empirically determined)
    if ( $rdb->{loc} > 0 ) {
        seek( $rdb->{fh}, $rdb->{loc}, 0 );
    }
    else {
        seek( $rdb->{fh}, 0, 2 );
    }

    return 1;
}

#pod =method rewind
#pod
#pod   $rdb->rewind;
#pod
#pod Rewind the file back to the first data position (i.e., after
#pod the header).  Obviously this only works if the file is truly
#pod a file, and not a pipe.
#pod
#pod =cut

sub rewind {
    1 == @_ or Carp::croak 'usage: $rdb->rewind()';

    my ( $rdb ) = @_;

    if ( defined $rdb->{start} ) {
        # if $rdb->{loc} is -1, at EOF (empirically determined)
        if ( $rdb->{start} > 0 ) {
            seek( $rdb->{fh}, $rdb->{start}, 0 );
        }
        else {
            seek( $rdb->{fh}, 0, 2 );
        }
    }
    $rdb->{loc} = $rdb->{start};

    return;
}



#pod =method write
#pod
#pod   $rdb-> write( @data | \@data | \%data )
#pod
#pod Write the passed data to the rdb file. If an array or a
#pod reference to an array is passed, it must have the correct number
#pod of columns, and must be in the same order as the columns in the
#pod rdb file.  If a reference to a hash is passed, the data are
#pod extracted from the hash.
#pod
#pod =cut

sub write {
    @_ >= 2 or Carp::croak 'usage : $rdb->write( @data|\@data|\%data )';

    my $rdb = shift;
    my $fh  = $rdb->{'fh'};

    # turn off warnings here; don't care about undefined values, as
    # they will just get mapped to empty (undefined) columns.

    local $^W = 0;

    $rdb->write_hdr unless $rdb->{hdr_written};

    if ( @_ > 1 or !ref( $_[0] ) ) {
        # must be an array.  assume it's in the same order as the columns
        Carp::croak( 'RDB::write -> incorrect number of data elements: got ',
            scalar @_, ' expected ', $rdb->{ncols} )
          if @_ != $rdb->{ncols};
        print $fh join( "\t", @_ ), "\n";
    }
    elsif ( ref( $_[0] ) eq 'HASH' ) {
        # must be a hash.
        print $fh join( "\t", @{ $_[0] }{ @{ $rdb->{'col'} } } ), "\n";
    }
    elsif ( ref( $_[0] ) eq 'ARRAY' ) {
        # must be an array
        Carp::croak(
            'RDB::write -> incorrect number of data elements: got ',
            scalar @{ $_[0] },
            ' expected ', $rdb->{ncols} ) if @{ $_[0] } != $rdb->{ncols};

        print $fh join( "\t", @{ $_[0] } ), "\n";
    }
    $rdb->{loc} = tell( $rdb->{fh} );

    return;
}

#pod =method write_hdr
#pod
#pod   $rdb->write_hdr( [<I<filehandle>>] );
#pod
#pod Write the RDB header to the passed file handle, if present,
#pod or to the filehandle associated with the RDB object.  Header lines
#pod containing header variables will be updated with the most recent
#pod value.  New header variables are appended to the end of the header.
#pod B<write_hdr> is automatically called for you if you try to B<write>
#pod or B<close> the object.
#pod
#pod =cut

sub write_hdr {
    @_ >= 1 or Carp::croak 'usage : $rdb->write_hdr( [$fh] )';

    my ( $rdb, $fh ) = @_;

    return if $rdb->{hdr_written};

    $fh = $rdb->{fh} unless defined $fh;

    return unless defined $fh;

    $rdb->_write_hdr_comments( $fh );


    # turn off warnings here; don't care about undefined values, as
    # they are just attributes that haven't been defined.
    if ( $rdb->ncols ) {
        local $^W = 0;

        print $fh join( "\t", @{ $rdb->{'col'} } ), "\n";
        print $fh join(
            "\t",
            map {
                "$_->{width}$_->{type}$_->{align}"
                  . ( $_->{desc} ? " $_->{desc}" : '' )
            } @{ $rdb->{'defs'} }{ @{ $rdb->{'col'} } }
          ),
          "\n";
        $rdb->{loc} = tell( $rdb->{fh} );
    }
    $rdb->{hdr_written}++;

    return;
}

sub _write_hdr_comments {
    my $rdb = shift;
    my $fh  = shift;

    local $_;

    my %hdrvar_written;
    foreach ( @{ $rdb->{comments} } ) {
        # update header variables
        if ( /^(:\s*(\w+)\s*=\s*).*/ ) {
            print $fh "#$1", $rdb->{hdrvars}{$2}, "\n";
            $hdrvar_written{$2}++;
        }
        else {
            print $fh "#$_\n";
        }
    }

    # now write out header variables that weren't in the comments
    print $fh "#: $_ = ", $rdb->{hdrvars}{$_}, "\n"
      foreach grep { !$hdrvar_written{$_} } sort keys %{ $rdb->{hdrvars} };

    return;
}


sub DESTROY {
    my ( $rdb ) = @_;
    $rdb->close();

    return;
}

#pod =method add_comments
#pod
#pod   $rdb->add_comments( @comments );
#pod
#pod Append the passed list of comments to the header comment lines.  The
#pod comments should neither begin with a leading pound sign nor end with a
#pod newline.  This method doesn't add any leading white space to the
#pod comment, so you may wish to do that for the sake of legibility.  If
#pod the comment line defines a header variable, the first character must
#pod be a C<:>.  You can later change it's value with B<setvar()> or
#pod re-read it with B<getvar()>.
#pod
#pod =cut

sub add_comments {
    my $rdb = shift;

    for my $comment ( map { length $_ ? split /^/ : $_ } @_ ) {
        chomp $comment;
        push @{ $rdb->{comments} }, $comment;

        if ( $comment =~ /^:\s*(\w+)\s*=\s*(.*)/ ) {
            $rdb->{hdrvars}{$1} = $2;
        }

    }

    return;
}

#pod =method col
#pod
#pod   @cols = $rdb->col;
#pod
#pod This returns a list containing the names of the columns in the
#pod RDB table if the calling routine is expecting a list, otherwise
#pod it returns a reference to the list of columns.
#pod
#pod         @cols = $rdb->col;
#pod         $cols_ref = $rdb->col;
#pod
#pod =cut

sub col {
    my $rdb = shift;
    return wantarray ? @{ $rdb->{col} } : [ @{ $rdb->{col} } ];
}

#pod =method comments
#pod
#pod   @comments = $rdb->comments( [@comments] )
#pod
#pod This returns a list containing the header comment lines.  The
#pod leading pound signs and trailing newline are
#pod removed.
#pod
#pod         @comments = $rdb->comments;
#pod         $rdb->comments( @replacement_comments );
#pod
#pod B<comments> takes as an optional argument a list containing new
#pod comments.  These will I<replace> the existing ones.  This method
#pod doesn't add any leading white space to the comment, so you may wish to
#pod do that for the sake of legibility.  To delete the comments, pass
#pod it C<undef>:
#pod
#pod         $rdb->comments( undef );
#pod
#pod =cut

sub comments {
    my $rdb = shift;

    if ( 1 == @_ && !defined $_[0] ) {
        $rdb->{comments} = [];
    }
    elsif ( @_ ) {
        $rdb->{comments} = [];
        $rdb->add_comments( @_ );
    }

    return @{ $rdb->{comments} };
}

#pod =method defs
#pod
#pod   @defs = $rdb->defs( [$name | @names] );
#pod
#pod This method is deprecated.  Use B<type> instead.
#pod
#pod =cut

sub defs {
    my $rdb = shift;
    return $rdb->type( @_ );
}

#pod =method defn
#pod
#pod    %hash = $rdb->defn;
#pod    $defn = $rdb->den( $name );
#pod    $array =  $rdb->defn( @names );
#pod
#pod If called with no arguments, C<defn> returns a hash containing the
#pod column definitions, keyed off of the column names.  The optional arguments
#pod are names of columns for which to return a definition.  In a scalar context
#pod it returns the definition for the first argument; in an array context it
#pod returns an array of definitions.  If a column doesn't exist, its definition is
#pod given as the undefined value.
#pod
#pod A definition is returned as a hash reference.  The hash has keys
#pod C<type>, C<width>, C<align>, and C<desc>.  Don't change the contents
#pod of the hashes returned!!
#pod
#pod =cut

sub defn {
    my $rdb = shift;

    return @_
      ? (
        wantarray
        ? map { $rdb->{defs}{$_} } @_
        : $rdb->{defs}{ $_[0] } )
      : %{ $rdb->{defs} };
}


#pod =method fh
#pod
#pod   $fh = $rdb->fh;
#pod
#pod This returns the filehandle to which the RDB file is attached,
#pod or the undefined value if it hasn't yet been attached.
#pod To attach a filehandle to an existing B<RDB> object, use
#pod the C<RDB::open> method.
#pod
#pod
#pod =cut

sub fh {
    my $rdb = shift;
    return $rdb->{fh};
}

#pod =method file
#pod
#pod   $file = $rdb->file;
#pod
#pod returns the filename or handle passed to the C<new> or C<open> method.
#pod
#pod =cut

sub file {
    my $rdb = shift;
    return $rdb->{file};
}

#pod =method ncols
#pod
#pod   $ncols = $rdb->ncols;
#pod
#pod This returns the number of columns in the file.
#pod
#pod
#pod =cut

sub ncols {
    my $rdb = shift;
    return $rdb->{ncols};
}

#pod =method pos
#pod
#pod         %pos = $rdb->pos;
#pod         $pos_of_col = $rdb->pos( $col );
#pod         @pos{@cols} = $rdb->defs(@cols);
#pod
#pod If called with no arguments, C<pos> returns a hash relating the column
#pod names to their zero-indexed position, keyed off of the column names.
#pod The optional arguments
#pod are names of columns for which to return a position.  In a scalar context
#pod it returns the position for the first argument; in an array context it
#pod returns an array of positions.  If a column doesn't exist, its position
#pod is given as the undefined value.
#pod
#pod
#pod =cut

sub pos {
    my $rdb = shift;
    return
        @_
      ? wantarray
          ? map { $rdb->{'pos'}{$_} } @_
          : $rdb->{'pos'}{ $_[0] }
      : %{ $rdb->{'pos'} };
}

#pod =method vars
#pod
#pod   @names = $rdb->vars;
#pod
#pod Returns the names of the header variables.
#pod
#pod =cut

sub vars {
    @_ == 1 or Carp::croak 'usage : $rdb->vars;';

    my $rdb = shift;

    return keys %{ $rdb->{hdrvars} };
}


#pod =method getvar
#pod
#pod   $vlue = $rdb->getvar( $var );
#pod
#pod Returns the value of the header variable I<$var> if it exists.
#pod Otherwise it returns C<undef>.
#pod
#pod =cut

sub getvar {
    @_ == 2 or Carp::croak 'usage : $rdb->getvar( $var )';

    my ( $rdb, $var ) = @_;

    return $rdb->{hdrvars}{$var};
}

#pod =method getvars
#pod
#pod   $hash = $rdb->getvars;
#pod
#pod Returns a reference to an hash containing all of the header
#pod variables.
#pod
#pod =cut

sub getvars {
    @_ == 1 or Carp::croak 'usage : $rdb->getvars';

    my $rdb = shift;

    return { %{ $rdb->{hdrvars} } };
}

#pod =method setvar
#pod
#pod   $rdb->setvar( $var, $value );
#pod
#pod Set the header variable I<$var> to I<$value>.  The variable is
#pod created if it doesn't exist.
#pod
#pod =cut

sub setvar {
    @_ == 3 or Carp::croak 'usage : $rdb->setvar( $var, $value )';

    my ( $rdb, $var, $value ) = @_;

    chomp $value;
    Carp::croak( "RDB header values may not span multiple lines: $value\n" )
      if $value =~ /\n/;

    # if the variable doesn't yet exist, there isn't an existing
    # header comment line; we create one so that the order of
    # comments and variables is retained.  one day there will
    # be proper handling of things...

    $rdb->add_comments( ": $var = $value" )
      unless exists $rdb->{hdrvars}{$var};

    $rdb->{hdrvars}{$var} = $value;

    return;
}

#pod =method delvar
#pod
#pod   $rdb->delvar( $var );
#pod
#pod Delete the header variable I<$var>.  This also deletes the header comment
#pod line which defines it.
#pod
#pod =cut

sub delvar {
    @_ == 2 or Carp::croak 'usage : $rdb->delvar( $var )';

    my ( $rdb, $var ) = @_;
    delete $rdb->{hdrvars}{$var};

    # delete from the comments list
    $rdb->{comments}
      = [ grep { !( /^:\s*(\w+)\s*=/ && $1 eq $var ) } @{ $rdb->{comments} } ];

    return;
}



#pod =method type
#pod
#pod   $rdb->type( [ $col | @cols ] );
#pod
#pod   %types = $rdb->type;
#pod   $type_of_col = $rdb->type($col);
#pod   @types{@cols} = $rdb->type(@cols);
#pod
#pod If called with no arguments, C<type> returns a hash containing the
#pod column types, keyed off of the column names.  The optional arguments
#pod are names of columns for which to return a type.  In a scalar context
#pod it returns the type for the first argument; in an array context it
#pod returns an array of types.  If a column doesn't exist, its type is
#pod given as the undefined value.
#pod
#pod
#pod =cut

sub type {
    my $rdb = shift;

    return @_
      ? (
        wantarray
        ? map { $rdb->{defs}{$_}{type} } @_
        : $rdb->{defs}{ $_[0] }{type} )
      : map { ( $_, $rdb->{defs}{$_}{type} ) } @{ $rdb->{col} };
}

# Expression parsing stuff

use vars qw( %cmpop  %assignop %optype  %mathop %punct %resw
  %builtin %spec_var %spec_defs );

# comparison operators
%cmpop = (
    N => {
        'lt'   => '<',
        'le'   => '<=',
        'gt'   => '>',
        'ge'   => '>=',
        'eq'   => '==',
        'ne'   => '!=',
        'mat'  => '=~',
        'nmat' => '!~',
        # not really a comparison operator, but it works better here
        'subst' => 's',
    },
    S => {
        'lt'   => 'lt',
        'le'   => 'le',
        'gt'   => 'gt',
        'ge'   => 'ge',
        'eq'   => 'eq',
        'ne'   => 'ne',
        'mat'  => '=~',
        'nmat' => '!~',
        # not really a comparison operator, but it works better here
        'subst' => 's',
    } );

%assignop = (
    '='   => { op => '=',   type => '' },
    '-='  => { op => '-=',  type => 'N' },
    '+='  => { op => '+=',  type => 'N' },
    '*='  => { op => '*=',  type => 'N' },
    '/='  => { op => '/=',  type => 'N' },
    '<<=' => { op => '<<=', type => 'N' },
    '>>=' => { op => '>>=', type => 'N' },
    '%='  => { op => '%=',  type => 'N' },
    '.='  => { op => '.=',  type => 'S' },
    '=~'  => { op => '=~',  type => 'N' },
);

%mathop = map { $_, $_ } qw( - + * / << >> && || & ^ ? : % ** . <=> );

%punct = map { $_, $_ } ( '(', ')', ',', ';', '{', '}' );
$punct{'{'} = "{\n";
$punct{'}'} = "}\n";
$punct{';'} = ";\n";

# reserved words
%resw = (
    'or'     => '||',
    'and'    => '&&',
    'null'   => '""',
    'any'    => '$_',
    'if'     => 'if',
    'else'   => 'else',
    'elsif'  => 'elsif',
    'unless' => 'unless'
);

%builtin = (
    # builtin functions that take a numeric argument
    (
        map { $_ => { func => $_, type => 'N' } }
          qw( abs atan2 cos exp hex int log oct rand sin sqrt )
    ),

    # builtin functions that take a string argument
    ( map { $_ => { func => $_, type => 'S' } } qw( lc uc sprintf ) ) );

# special "force type" functions
$builtin{'_fX'} = { func => '', type => 'X' };
$builtin{'_fN'} = { func => '', type => 'N' };
$builtin{'_fS'} = { func => '', type => 'S' };

%spec_var  = ( _NR  => '$.' );
%spec_defs = ( '$.' => 'N' );

#pod =method expr
#pod
#pod   $rdb-> expr( I<reference> or I<string> );
#pod
#pod B<This is deprecated and will be removed in the future.>
#pod
#pod Given an rdb expression (see the B<rdb_expr> documentation), it
#pod returns a string which evaluates the expression.  If passed a
#pod reference, the reference must be to an array of tokens.  If passed a
#pod scalar, the scalar is parsed into tokens.  Note that the whitespace
#pod requirements for the passed scalar is the same as that in the
#pod B<rdb_expr> documentation.
#pod
#pod The returned expression assumes that an input row of the RDB
#pod table is available in the C<F> list.
#pod
#pod For example, given the following RDB table,
#pod
#pod   Name    Address Zip
#pod   S       S       N
#pod   me      here    20
#pod   you     there   30
#pod
#pod the function call
#pod
#pod   print $rdb->expr( 'Name eq "Bill plus jane" and Zip eq 3' );
#pod
#pod produces the following string:
#pod
#pod   $F[0] eq 'Bill plus jane' && $F[2] == 3
#pod
#pod =cut

sub expr {
    @_ >= 2
      or Carp::croak 'usage : $rdb->expr( \@args ) or $rdb->expr( $string )';
    my ( $rdb, $args ) = @_;
    my ( $defs ) = $rdb->{'defs'};
    my ( $pos )  = $rdb->{'pos'};

    my ( @args, $oper_type, $pcol, $poper, $expect_type );

    if ( !ref( $args ) ) {
        require Text::ParseWords;
        @args = Text::ParseWords::shellwords( $args );
        $args = \@args;
    }
    elsif ( 'ARRAY' ne ref( $args ) ) {
        Carp::croak 'usage : $rdb->expr( \@args ) or $rdb->expr( $string )';
    }


    for my $arg ( @$args ) {
        if ( $$defs{$arg} ) {
            $poper = "COL";
            $pcol  = $arg;    # prev col name
            $oper_type = ( $$defs{$arg} =~ /(\S+)/i && $1 =~ /N/i ) ? 'N' : 'S';
            $arg       = '$F[' . $$pos{$arg} . ']';
            next;
        }

        if ( $arg =~ /^_/ && defined $spec_var{$arg} ) {
            $arg = $spec_var{$arg};

            # numerical data flag
            $oper_type
              = ( $spec_defs{$arg} =~ /(\S+)/ && $1 =~ /N/i ) ? 'N' : 'S';
            next;
        }

        # cmp oper
        if ( $cmpop{N}{$arg} || $cmpop{S}{$arg} ) {
            $poper = "CMP";    # prev oper, for next cycle

            if ( $arg =~ /mat$/ ) {
                Carp::carp(
                    "Warning, Pattern Match on numeric column ($pcol)\n" )
                  if $oper_type eq 'N';
                $poper = "CMPM";
            }

            $arg = $cmpop{ $oper_type ? $oper_type : 'N' }{$arg};

            next;
        }

        # reserved word
        if ( $resw{$arg} ) {
            $arg   = $resw{$arg};
            $poper = "RES";
            next;
            $oper_type = '';
        }

        if ( $builtin{$arg} ) {
            $oper_type = $builtin{$arg}->{type};
            $poper     = "FUNC";
            $arg       = $builtin{$arg}->{func};
            next;
        }

        if ( $assignop{$arg} ) {
            $expect_type = $assignop{$arg}->{type};
            $arg         = $assignop{$arg}->{op};
            # make up for ambiguity of '=' op by using previous operand type,
            # if available
            $expect_type = $oper_type if $arg eq '=';
            $poper       = "OP";
            $oper_type   = '';
            next;
        }

        if ( $punct{$arg} ) {
            $arg = $punct{$arg};
            next;
        }

        if ( $mathop{$arg} ) {
            $arg       = $mathop{$arg};
            $poper     = "OP";
            $oper_type = '';
            next;
        }

        # if it begins with a '$', assume it's a local Perl variable,
        # and leave it alone
        if ( '$' eq substr( $arg, 0, 1 ) )    #'
        {
            $poper     = "VAL";
            $oper_type = '';
            next;
        }

        # numeric data
        if ( $oper_type eq 'N' ) {
            # it's supposed to be numerical, but if it's not, make it a string
            if ( _is_numeric( $arg ) ) {
                _handle_numeric( \$arg, \$poper );
            }
            else {
                _handle_string( \$arg, \$poper );
            }
            $oper_type = '';
            next;
        }

        # string data stick it in quotes
        if ( $oper_type eq 'S' ) {
            _handle_string( \$arg, \$poper );
            $oper_type = '';
            next;
        }

        # as-is
        if ( $oper_type eq 'X' ) {
            next;
        }

        # this data didn't follow a comparison operator (probably
        # from an assignment expression).  If it's a number, assume
        # it's not a string.
        # first ensure there's not a string assignment operator
        if ( $expect_type ne 'S' && _is_numeric( $arg ) ) {
            _handle_numeric( \$arg, \$poper );
            $oper_type = 'N';
            next;
        }

        # everything else is assumed to be a string
        _handle_string( \$arg, \$poper );
        $poper     = "VAL";
        $oper_type = 'S';

    }

    return join( ' ', @args );
}

sub _is_numeric {
    my ( $arg ) = @_;

    return $arg =~ /^[+-]?(\d+[.]?\d*|[.]\d+)([eE][+-]?\d+)?$/;
}

sub _handle_string {
    my ( $arg, $poper ) = @_;
    # if the last operand was a match, don't quote it
    if ( $$poper ne 'CMPM' ) {
        # if the first character in the string is a `\',
        # it's been put there to escape the second character.
        # this isn't a means of escaping control characters, but
        # to avoid recognition as a column name or local variable
        # we need to remove it
        $$arg =~ s/^[\\]//;
        $$arg =~ s/[\\]?([\'])/\\$1/g;
        $$arg = "'$$arg'";
    }
    $$poper = "VAL";

    return;
}

sub _handle_numeric {
    my ( $arg, $poper ) = @_;

    # remove any leading zeros as they'll be interpreted
    # as octal
    $$arg =~ s/^0+([1-9]+)/$1/;
    $$poper = "VAL";

    return;

}


1;

#
# This file is part of RDB
#
# This software is Copyright (c) 2017 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

=pod

=head1 NAME

RDB - object methods for dealing with rdb files

=head1 VERSION

version 1.53

=head1 SYNOPSIS

  use RDB;

  $rdb = new RDB;
  $rdb->open( 'foo.rdb' ) || die;

  $rdb = new RDB 'foo.rdb' or die;

  $rdb = new RDB \*STDIN or die;

  $rdb = new RDB ( 'name' => 'S', 'id' => 'N' );

  $rdb->init( );
  $rdb->init( 'name' => 'S', 'id' => 'N' );

  $rdb->add_col( 'slap' => 'S', 'gurgle' => 'N' );
  $rdb->add_col( $other_rdb );

  $rdb->delete_col( 'gurgle' );

  @defs = ( 'name' => 'S', 'id' => 'N' );
  $rdb->init( \@defs );

  $rdb->init( $other_rdb );

  $rdb->rewind;

  $rdb->bind( { col1 => \$col1, col2 => \$col2 } );
  while ( $rdb->read( ) ) { print $col1, $cols, "\n"; }

  while( $rdb->read( \%data ) ) { ... }
  while( $rdb->read( \@data ) ) { ... }

  while( $verbatim_line = $rdb->read_line ) { print $verbatim_line };

  $rdb->write_hdr( );
  $rdb->write_hdr( \*STDOUT );

  $rdb->write( \%data );
  $rdb->write( \@data );
  $rdb->write( @data );

  $rdb->set( \%attr );
  $rdb->set( { AlwaysBind => 1 } );

  @header_var_names = $rdb->vars;

  $foo = $rdb->getvar( 'foo' );
  $rdb->setvar( foo => 33 );
  $hdrvars = $rdb->getvars;
  print $hdrvars->{foo};
  $rdb->delvar ( 'foo' );

=head1 DESCRIPTION

This module eases use of RDB data files.  It creates RDB objects
which contain the necessary information for interpreting and
manipulating RDB files.

=head1 METHODS

=head2 new

  $rdb = RDB->new( [$data, [$mode] | \@defs ] );

Create an B<RDB> object, optionally attaching it to a data source.
If C<$data> (and optionally C<$mode> is specified, the B<open> method is invoked.
If C<@defs> is passed, the B<init> method is invoked.

C<$data> may be one of the following:

=over

=item I<filename>

The name of a file (a scalar)

=item I<scalar ref>

A string containing RDB formatted data.  Pass a I<reference> to the string.

=item I<file handle>

An existing, opened file.

=back

C<$mode> defaults to C<< < >>.

=head2 bind

   $rdb->bind( \%bindhash [, \%attrs ] )

B<bind> simplifies the processing of rdb files by allowing the
automatic assignment of values read from the rdb file to Perl
variables or arrays. Each time that the read method is called with no
arguments, it will update the variables specified in preceding calls
to C<bind>.  B<bind> takes a hash of columns to be bound; the keys are
the column names, their values are references to either scalars or
arrays.  In the former case, the scalar will be assigned the column's
value.  In the latter case, the column's value is pushed onto the end
of the array.  ( Note that the argument to bind is a hash just to
enforce the correct number of items.)  For example,

        $rdb->bind( { col1 => \$col1, col2 => \$col2 } );
        while ( $rdb->read( ) )
        {
          print "$col1, $cols\n";
        }

Or, using arrays,

        my ( @col1, @col2 );
        $rdb->bind( { col1 => \@col1, col2 => \@col2 } );
        1 while ( $rdb->read( ) );
        for( $i = 0 ; $i < @col1 ; $i++ )
        {
          print $col1[$i], ' ', $col2[$i], "\n";
        }

If the same column is specified in I<succeeding> calls to B<bind>, the
new binding will override the previous binding.

However, if the same column should be bound to multiple variables, the
C<Override> attribute may be reset using the second argument to
B<bind>:

        $rdb->bind( { col1 => \$col1, col2 => \$col2 } );
        $rdb->bind( { col1 => \$col1_copy }, { Override => 0 } );

The column C<col1> will now be written to both C<$col1> and C<$col1_copy>.

=head2 close

explicitly close an rdb file.  This usually need not be called, as the
file will be closed when the RDB object is destroyed.

=head2 init

  $rdb-> init( @defs | \@defs | $rdb );

Initialize the rdb object with a set of columns.  A column is
specified by both a name and a definition.  Definitions technically
consist of four parts: the column name, it's type, output alignment,
and description.  The latter are optional and are usually omitted.
Column types are one of C<N>, C<S>, or C<M>, for numeric, string, and
month data.  Alignment is one of C<<> or C<>>.

B<init> is passed either an array (or list), an array reference, or a
reference to another B<RDB> object.  In the latter case, the column
definitions of the other object are duplicated.  In the former cases,
the array must contain column name and definition pairs.

The definition may take any of the following forms:

=over 8

=item *

If the definition is a scalar, it should be the column type:

  $rdb->init( c1 => 'N' );

=item *

if the definition is a hash reference, it should have at least
the key C<type>, and may optionally have the keys C<width>, C<align>,
or C<desc>

  $rdb->init( c1 => { type => 'N', width => 3, align => '<',
                      desc => 'This column is meaningless } );

=item *

if the definition is an array reference, it may have up to four
elements; the type, width, alignment, and description, in order.
The last three are optional.

  $rdb->init( c1 => [ 'N', 3, '<', 'This column is meaningless' ] );

=back

Any of these forms may be mixed:

  $rdb->init( c1 => 'N',
              c2 => [ 'N', 32 ],
              c3 => { type => 'N', desc => 'What A Nice Column' } );

=head2 init_tpl

  $rdb->init_tpl( $file_name | \$tpl_string )

Initialize an RDB object from an RDB header template. If the passed
argument is a scalar, it should contain the name of a file containing
the template.  If it's a reference it should be a reference to a
scalar containing the template.  An RDB header template is description
of the header in the following format.

Each column is specified on a separate line, and contains up to
four white space delimited fields:

=over 8

=item 1

an optional field containing the column's zero based index.  If not specified,
the ordering of the field in the template is used.  For example,

   fee S
   fie N
   fo  N
   fum N

is equivalent to

   0 fee S
   1 fie N
   2 fo  N
   3 fum N

Be careful when mixing lines with and without an index:

     fee S
   2 fie N
     fo  N
     fum N

is equivalent to

   0 fee S
   2 fie N
   2 fo  N
   3 fum N

which will result in an error.  Indices must be unique.

There's a further degeneracy which must be avoided:

   3 N S

Is that an index of C<3>, a name of C<N> and a type of C<S>, or is
that a name of C<3>, a type of C<N> and a description of C<S>?
It is parsed as the former.  To get the latter interpretation,
you'll have to include an index field.

=item 2

the column name.  it may appear in quotes.

=item 3

the column type.  it may include the column width as a prefix

=item 4

an optional column description

=back

Comment lines may be present, and are indicated by a leading C<#> character.

For example,

  # P-to-H Decenter parameters derived from XRCF HSI off-axis images
  # (single shell); used pitch=0, yaw=-20 arcmin data.
  #
   0               fee  6S      what i get paid
   1               fie 10N      upon you
   2               fo  10N      fight or no?
   3               fum  9N      ble

=head2 write_tpl

  $rdb->write_tpl( $filename | $fh);

Write an RDB template for the current RDB object.  The argument may be
a scalar, it which case it should contain the name of a file to which
to write the template, or a filehandle.

=head2 add_col

  $rdb->add_col( @defs | \@defs | $rdb);

Add new columns to the rdb object.  See the description of the
B<init()> method for the specification of the column names and
definitions.  Existing columns are not duplicated; their definitions
are changed to the passed type.

=head2 delete_col

  $rdb-> delete_col( @cols );

delete the specified columns from the object.  This is only applicable
to RDB files open for writing, and only before the RDB header has
been written out.

        $rdb->delete_col( 'a', 'b' );

=head2 set

  $rdb-> set( \%attr );

B<set> specifies the values of various attributes for the object.
The passed reference should point to a hash which may contain
the following keys:

=over

=item AlwaysBind

If this is set, e.g.,

  $rdb->set( { AlwaysBind => 1} )

if B<RDB::bind> has been called to set up bindings between columns and
Perl variables, the Perl variables will always be updated, regardless of which
form of B<RDB::read> is called.

=back

=head2 open

  $rdb->open( $data, [ $mode ] );

Attach the object to a data source.

C<$data> may be one of the following:

=over

=item I<filename>

The name of a file (a scalar)

=item I<scalar ref>

A string containing RDB formatted data.  Pass a I<reference> to the string.

=item I<file handle>

An existing, opened file.

=back

Modes are the standard Perl-ish ones (see the Perl open command).  If
the mode is read only or read/write, it reads and parses the RDB
header.  C<$mode> defaults to C<< < >>.

It returns C<undef> upon error.

=head2 read

  $rdb->read( [ %data | @data ] );

Read in the next line from the rdb database, storing the columns into
either a hash keyed off of the column names (if passed a reference to
the hash), an array (if passed a reference to the array), or into
scalars specified by previous calls to the C<bind> method (if C<read>
is called with no arguments). It returns the undefined value upon end
of file.  It does not check to ensure that there are enough columns in
the input.  For example:

        $rdb->read(\%data);
        print "foo = $data{foo}\n";

        $rdb->read(\@data);
        print "The first column has value $data[0]\n";

        $rdb->bind( { foo => \$foo } );
        $rdb->read();
        print "Foo = $foo\n";

=head2 read_line

  $rdb->read_line;

C<read_line> reads a line from the rdb file without parsing it (even
to chop off the end).  It returns C<undef> upon end of file.

=head2 rename

  $rdb->rename( \%renamehash );

B<rename> is passed a hash of columns to be renamed; the keys are
the old column names, their values are the new names.  It's a hash
just to enforce the correct number of items.  For example,

        $rdb->rename( { oldcol => 'newcol', foocol => 'boocol' } );

=head2 reopen

  $rdb-> reopen;

B<reopen> reopens a file that has previously been opened and closed,
positioning the filepointer to where it was before it was closed.  It
retains the previous access mode.  It does not reopen files passed to
the original call of rdb::open as references.  It returns the
undefined value upon error.

=head2 rewind

  $rdb->rewind;

Rewind the file back to the first data position (i.e., after
the header).  Obviously this only works if the file is truly
a file, and not a pipe.

=head2 write

  $rdb-> write( @data | \@data | \%data )

Write the passed data to the rdb file. If an array or a
reference to an array is passed, it must have the correct number
of columns, and must be in the same order as the columns in the
rdb file.  If a reference to a hash is passed, the data are
extracted from the hash.

=head2 write_hdr

  $rdb->write_hdr( [<I<filehandle>>] );

Write the RDB header to the passed file handle, if present,
or to the filehandle associated with the RDB object.  Header lines
containing header variables will be updated with the most recent
value.  New header variables are appended to the end of the header.
B<write_hdr> is automatically called for you if you try to B<write>
or B<close> the object.

=head2 add_comments

  $rdb->add_comments( @comments );

Append the passed list of comments to the header comment lines.  The
comments should neither begin with a leading pound sign nor end with a
newline.  This method doesn't add any leading white space to the
comment, so you may wish to do that for the sake of legibility.  If
the comment line defines a header variable, the first character must
be a C<:>.  You can later change it's value with B<setvar()> or
re-read it with B<getvar()>.

=head2 col

  @cols = $rdb->col;

This returns a list containing the names of the columns in the
RDB table if the calling routine is expecting a list, otherwise
it returns a reference to the list of columns.

        @cols = $rdb->col;
        $cols_ref = $rdb->col;

=head2 comments

  @comments = $rdb->comments( [@comments] )

This returns a list containing the header comment lines.  The
leading pound signs and trailing newline are
removed.

        @comments = $rdb->comments;
        $rdb->comments( @replacement_comments );

B<comments> takes as an optional argument a list containing new
comments.  These will I<replace> the existing ones.  This method
doesn't add any leading white space to the comment, so you may wish to
do that for the sake of legibility.  To delete the comments, pass
it C<undef>:

        $rdb->comments( undef );

=head2 defs

  @defs = $rdb->defs( [$name | @names] );

This method is deprecated.  Use B<type> instead.

=head2 defn

   %hash = $rdb->defn;
   $defn = $rdb->den( $name );
   $array =  $rdb->defn( @names );

If called with no arguments, C<defn> returns a hash containing the
column definitions, keyed off of the column names.  The optional arguments
are names of columns for which to return a definition.  In a scalar context
it returns the definition for the first argument; in an array context it
returns an array of definitions.  If a column doesn't exist, its definition is
given as the undefined value.

A definition is returned as a hash reference.  The hash has keys
C<type>, C<width>, C<align>, and C<desc>.  Don't change the contents
of the hashes returned!!

=head2 fh

  $fh = $rdb->fh;

This returns the filehandle to which the RDB file is attached,
or the undefined value if it hasn't yet been attached.
To attach a filehandle to an existing B<RDB> object, use
the C<RDB::open> method.

=head2 file

  $file = $rdb->file;

returns the filename or handle passed to the C<new> or C<open> method.

=head2 ncols

  $ncols = $rdb->ncols;

This returns the number of columns in the file.

=head2 pos

        %pos = $rdb->pos;
        $pos_of_col = $rdb->pos( $col );
        @pos{@cols} = $rdb->defs(@cols);

If called with no arguments, C<pos> returns a hash relating the column
names to their zero-indexed position, keyed off of the column names.
The optional arguments
are names of columns for which to return a position.  In a scalar context
it returns the position for the first argument; in an array context it
returns an array of positions.  If a column doesn't exist, its position
is given as the undefined value.

=head2 vars

  @names = $rdb->vars;

Returns the names of the header variables.

=head2 getvar

  $vlue = $rdb->getvar( $var );

Returns the value of the header variable I<$var> if it exists.
Otherwise it returns C<undef>.

=head2 getvars

  $hash = $rdb->getvars;

Returns a reference to an hash containing all of the header
variables.

=head2 setvar

  $rdb->setvar( $var, $value );

Set the header variable I<$var> to I<$value>.  The variable is
created if it doesn't exist.

=head2 delvar

  $rdb->delvar( $var );

Delete the header variable I<$var>.  This also deletes the header comment
line which defines it.

=head2 type

  $rdb->type( [ $col | @cols ] );

  %types = $rdb->type;
  $type_of_col = $rdb->type($col);
  @types{@cols} = $rdb->type(@cols);

If called with no arguments, C<type> returns a hash containing the
column types, keyed off of the column names.  The optional arguments
are names of columns for which to return a type.  In a scalar context
it returns the type for the first argument; in an array context it
returns an array of types.  If a column doesn't exist, its type is
given as the undefined value.

=head2 expr

  $rdb-> expr( I<reference> or I<string> );

B<This is deprecated and will be removed in the future.>

Given an rdb expression (see the B<rdb_expr> documentation), it
returns a string which evaluates the expression.  If passed a
reference, the reference must be to an array of tokens.  If passed a
scalar, the scalar is parsed into tokens.  Note that the whitespace
requirements for the passed scalar is the same as that in the
B<rdb_expr> documentation.

The returned expression assumes that an input row of the RDB
table is available in the C<F> list.

For example, given the following RDB table,

  Name    Address Zip
  S       S       N
  me      here    20
  you     there   30

the function call

  print $rdb->expr( 'Name eq "Bill plus jane" and Zip eq 3' );

produces the following string:

  $F[0] eq 'Bill plus jane' && $F[2] == 3

=head1 SEE ALSO

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2017 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut

__END__


#pod =head1 SYNOPSIS
#pod
#pod   use RDB;
#pod
#pod   $rdb = new RDB;
#pod   $rdb->open( 'foo.rdb' ) || die;
#pod
#pod   $rdb = new RDB 'foo.rdb' or die;
#pod
#pod   $rdb = new RDB \*STDIN or die;
#pod
#pod   $rdb = new RDB ( 'name' => 'S', 'id' => 'N' );
#pod
#pod   $rdb->init( );
#pod   $rdb->init( 'name' => 'S', 'id' => 'N' );
#pod
#pod   $rdb->add_col( 'slap' => 'S', 'gurgle' => 'N' );
#pod   $rdb->add_col( $other_rdb );
#pod
#pod   $rdb->delete_col( 'gurgle' );
#pod
#pod   @defs = ( 'name' => 'S', 'id' => 'N' );
#pod   $rdb->init( \@defs );
#pod
#pod   $rdb->init( $other_rdb );
#pod
#pod   $rdb->rewind;
#pod
#pod   $rdb->bind( { col1 => \$col1, col2 => \$col2 } );
#pod   while ( $rdb->read( ) ) { print $col1, $cols, "\n"; }
#pod
#pod   while( $rdb->read( \%data ) ) { ... }
#pod   while( $rdb->read( \@data ) ) { ... }
#pod
#pod   while( $verbatim_line = $rdb->read_line ) { print $verbatim_line };
#pod
#pod   $rdb->write_hdr( );
#pod   $rdb->write_hdr( \*STDOUT );
#pod
#pod   $rdb->write( \%data );
#pod   $rdb->write( \@data );
#pod   $rdb->write( @data );
#pod
#pod   $rdb->set( \%attr );
#pod   $rdb->set( { AlwaysBind => 1 } );
#pod
#pod   @header_var_names = $rdb->vars;
#pod
#pod   $foo = $rdb->getvar( 'foo' );
#pod   $rdb->setvar( foo => 33 );
#pod   $hdrvars = $rdb->getvars;
#pod   print $hdrvars->{foo};
#pod   $rdb->delvar ( 'foo' );
#pod
#pod =head1 SEE ALSO
#pod
#pod =head1 DESCRIPTION
#pod
#pod This module eases use of RDB data files.  It creates RDB objects
#pod which contain the necessary information for interpreting and
#pod manipulating RDB files.
