#ifndef dvm3_rotmat_h_INCLUDED
#define dvm3_rotmat_h_INCLUDED

// File:    dvm3_rotmat.h
// Author:  Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of dvm3
 *
 * dvm3 is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * dvm3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <dvm3/dvm3_matrix.h>           // dvm3_Matrix
#include <cfloat>                       // DBL_EPSILON

//########################################################################
// dvm3_RotMat
//########################################################################

/** 
 * \class dvm3_RotMat dvm3_rotmat.h <dvm3/dvm3_rotmat.h>
 * 
 * A class representing 3x3 rotation matrix of doubles.
 * dvm3_RotMat is a derived class of dvm3_Matrix.  Constructors and
 * initialization methods are added to construct orthogonal 3x3 matrices.
 *
 * This is intended as a small component which can be inexpensively created
 * on the stack; the constructors and destructor do not make use of
 * dynamic allocation.  The dvm3_RotMat default constructor initializes
 * the matrix to a unit matrix (i.e., no rotation). </p>
 *
 * Suppose we are considering two frames, an initial standard frame,
 * S, and a body frame, B.  Let <b>e</b><sub>S,i</sub>, i = 0, 1, 2, denote 
 * the i'th basis vector for frame S, and <b>e</b><sub>B,i</sub>, 
 * i = 0, 1, 2, the corresponding basis vectors in frame B.  The rotation
 * matrix from S to B can be written as 
 * <br> <center>
 *    <b>R</b><sub>i,j</sub> 
 *         = <b>e</b><sub>B,i</sub> . <b>e</b><sub>S,j</sub>
 * </center></br>
 * Note that this has two particularly simple interpretations:
 * <OL><li>
 * the column vectors of <b>R</b> consist of the direction
 * cosines of the <b>e</b><sub>S</sub> vectors in the
 * <b>e</b><sub>B</sub> frame</li>
 * <li>
 * the row vectors consist of the direction cosines of the 
 * <b>e</b><sub>B</sub> vectors in the <b>e</b><sub>S</sub> frame.</li>
 * </OL> </p>
 */
class dvm3_RotMat
  : public dvm3_Matrix
{
public:

  /**
   * \enum dvm3_RotMat::EEulerAngleType
   * Enumeration specifying the interpretation of the Euler angles.
   *
   * <ul>
   *   <li>EOSAC_az_el_clock: variant "xyz" Euler angles:
   *       <ul>
   *       <li> <b>angle0 = azimuth</b>, a rotation about the current Y axis.
   *            Positive azimuth takes Z towards X. </li>
   *       <li> <b>angle1 = elevation</b>, a rotation about the current X axis.
   *            Positive elevation takes Z towards Y. </li>
   *       <li> <b>angle2 = clocking</b>, a rotation about the current Z axis.
   *            Positive clocking takes X towards Y. </li>
   *       </ul>
   *       The body orientation relative to the external frame is formed
   *       as azimuth followed by elevation followed by clocking.
   *    </li>
   *   <li>EX_convention: "xyz" Euler angles, see
   *     <a href="http://mathworld.wolfram.com/EulerAngles.html">mathworld</a>:
   *       <ul>
   *       <li> <b>angle0</b>, a rotation about the current Z axis.
   *            Positive phi takes X towards Y. </li>
   *       <li> <b>angle1</b>, a rotation about the current X axis.
   *            Positive theta takes Y towards Z. </li>
   *       <li> <b>angle2</b>, a rotation about the current Z axis(again).
   *            Positive psi takes X towards Y. </li>
   *       </ul>
   *    </li>
   *   <li>EY_convention: "xyz" Euler angles, see 
   *     <a href="http://mathworld.wolfram.com/EulerAngles.html">mathworld</a>:
   *       <ul>
   *       <li> <b>angle0</b>, a rotation about the current Z axis.
   *            Positive phi takes X towards Y. </li>
   *       <li> <b>angle1</b>, a rotation about the current Y axis.
   *            Positive theta takes Z towards X. </li>
   *       <li> <b>angle2</b>, a rotation about the current Z axis(again).
   *            Positive psi takes X towards Y. </li>
   *       </ul>
   *    </li>
   *    </ul>
   */
  enum EEulerAngleType { EOSAC_az_el_clock = 0, 
                         EX_convention = 1,  
                         EY_convention = 2 };

  /**
   * \enum dvm3_RotMat::EInputVecPair
   * Enumeration specifying the interpretation of a pair of input vectors.
   *   <p>
   *   <ul>
   *   <li>E_XY: v0 is "x-like", v1 is "y-like".</li>
   *   <li>E_YZ: v0 is "y-like", v1 is "z-like".</li>
   *   <li>E_ZX: v0 is "z-like", v1 is "x-like".</li>
   *   <li>E_XZ: v0 is "x-like", v1 is "z-like".</li>
   *   <li>E_ZY: v0 is "z-like", v1 is "y-like".</li>
   *   <li>E_YX: v0 is "y-like", v1 is "x-like".</li>
   *   </ul>
   */
  enum EInputVecPair { E_XY, E_YZ, E_ZX, E_XZ, E_ZY, E_YX };

  /**
   * Do-nothing destructor
   */
 ~dvm3_RotMat();

  /**
   * Default constructor; a unit matrix is generated by default.
   */
  dvm3_RotMat();

  /**
   * Copy constructor.
   *
   * @param m     rotation matrix to copy
   */
  dvm3_RotMat( dvm3_RotMat const& m );

  /**
   * Constructor.
   *
   * Each row is a coordinate unit vector.
   * The vectors are assumed to be orthonormal; an error
   * is issued and exit(1) is called if the vectors are
   * not orthonormal to within tolerance tol.  If the
   * vectors are indeed orthonormal within tolerance tol, 
   * they are further explicitly orthonormalized
   * by a series of cross products and scalings; consequently
   * the matrix rows may differ slightly from the input 
   * x, y, z dvm3_Vector's.
   *
   * @param x     x row
   * @param y     y row
   * @param z     z row
   * @param tol   tolerance for orthonormality
   */
  dvm3_RotMat( dvm3_Vector const& x,
               dvm3_Vector const& y, 
               dvm3_Vector const& z,
               double const tol = 100.0 * DBL_EPSILON );

  /**
   * Constructor.
   *
   * Each row is a coordinate unit vector.
   * The vectors are assumed to be orthonormal; an error
   * is issued and exit(1) is called if the vectors are
   * not orthonormal to within tolerance tol.  If the
   * vectors are indeed orthonormal within tolerance tol, 
   * they are further explicitly orthonormalized
   * by a series of cross products and scalings; consequently
   * the matrix rows may differ slightly from the input 
   * x, y, z dvm3_Vector's.
   *
   * @param x     x row
   * @param y     y row
   * @param z     z row
   * @param tol   tolerance for orthonormality
   */
  dvm3_RotMat( double const x[],
               double const y[],
               double const z[],
               double const tol = 100.0 * DBL_EPSILON );

  /**
   * Constructor.
   *
   * Construct a rotation matrix based on 3 Euler angles (in radians).
   * This rotation matrix describes the rotation <em>from</em> an external 
   * system coordinate <em>to</em> a coordinate system attached to
   * a rigid body.
   * <p>
   * A body initially has its coordinate system aligned with the
   * standard external coordinates.  The body is rotated to its final
   * orientation by performing a series of rotations, each rotation about
   * a current body axis.  That is<ul>
   * <li>  rotation by angle0 about axis a, followed by</li>
   * <li>  rotation by angle1 about axis b, followed by</li>
   * <li>  rotation by angle2 about axis c</li>
   * </ul>
   * The specification of angle0, angle1, and angle2 and axes a, b, and c
   * are determined by the enumerated constant <code>angle_type.
   * <p>
   * Because we want a transformation <em>from</em> the space coordinates
   * <em>to</em> the body coordinates, the rotation matrix components
   * actually correspond to the inverse of the transformation which
   * generated the rigid body position, i.e.,<ul>
   * <li>  rotation by -angle2 about axis c, followed by</li>
   * <li>  rotation by -angle1 about axis b, followed by</li>
   * <li>  rotation by -angle0 about axis a</li>
   * </ul>
   * <p>
   * Because of rotation matrices are orthogonal, the inverse
   * motion (from the body-centered coordinate system back to
   * the standard coordinate system) is simply the transpose of the
   * rotation matrix constructed by this constructor.</p>
   *
   * @param angle0      1st Euler angle
   * @param angle1      2nd Euler angle
   * @param angle2      3rd Euler angle
   * @param angle_type  type of Euler angles to use
   */
  dvm3_RotMat( double angle0, double angle1, double angle2,
               EEulerAngleType angle_type = EOSAC_az_el_clock );

  /**
   * Constructor.
   *
   * Construct a rotation matrix based on two (non-colinear) vectors.
   * The rotation matrix is constructed by sequential application 
   * cross products.
   * <p>
   * If type == E_XY, the vectors passed in are assumed to lie in the 
   * x-y plane with v0 an x-like vector and v1 a y-like vector;
   * e_x is v0.  e_z is constructed from e_x cross v1.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YZ, the vectors passed in are assumed to lie in the
   * y-z plane with v0 an y-like vector and v1 a z-like vector;
   * e_y is v0.  e_x is constructed from e_y cross v1.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZX, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an z-like vector and v1 a x-like vector;
   * e_z is v0.  e_y is constructed from e_z cross v1.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YX, the vectors passed in are assumed to lie in the
   * x-y plane with v0 a y-like vector and v1 an x-like vector;
   * e_y is v0.  e_z is constructed from v1 cross e_y.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZY, the vectors passed in are assumed to lie in the
   * y-z plane with v0 a z-like vector and v1 a y-like vector;
   * e_z is v0.  e_x is constructed from v1 cross e_z.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_XZ, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an x-like vector and v1 a z-like vector;
   * e_x is v0.  e_y is constructed from v1 cross e_x.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   *
   * @param v0     1st vector
   * @param v1     2nd vector
   * @param type   type of input vectors
   */
  dvm3_RotMat( dvm3_Vector const& v0,
               dvm3_Vector const& v1,
               EInputVecPair      type
            );

  /**
   * Constructor.
   *
   * Construct a rotation matrix based on two (non-colinear) vectors.
   * The rotation matrix is constructed by sequential application 
   * cross products.
   * <p>
   * If type == E_XY, the vectors passed in are assumed to lie in the 
   * x-y plane with v0 an x-like vector and v1 a y-like vector;
   * e_x is v0.  e_z is constructed from e_x cross v1.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YZ, the vectors passed in are assumed to lie in the
   * y-z plane with v0 an y-like vector and v1 a z-like vector;
   * e_y is v0.  e_x is constructed from e_y cross v1.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZX, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an z-like vector and v1 a x-like vector;
   * e_z is v0.  e_y is constructed from e_z cross v1.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YX, the vectors passed in are assumed to lie in the
   * x-y plane with v0 a y-like vector and v1 an x-like vector;
   * e_y is v0.  e_z is constructed from v1 cross e_y.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZY, the vectors passed in are assumed to lie in the
   * y-z plane with v0 a z-like vector and v1 a y-like vector;
   * e_z is v0.  e_x is constructed from v1 cross e_z.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_XZ, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an x-like vector and v1 a z-like vector;
   * e_x is v0.  e_y is constructed from v1 cross e_x.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   *
   * @param v0     1st vector
   * @param v1     2nd vector
   * @param type   type of input vectors
   */
  dvm3_RotMat( double const v0[],
               double const v1[],
               EInputVecPair      type
            );

  /**
   * Initialize a rotation matrix based on 3 Euler angles (in radians).
   * This rotation matrix describes the rotation <em>from</em> an external
   * system coordinate <em>to</em> a coordinate system attached to
   * a rigid body.
   * <p>
   * A body initially has its coordinate system aligned with the
   * standard external coordinates.  The body is rotated to its final
   * orientation by performing a series of rotations, each rotation about
   * a current body axis.  That is<ul>
   * <li>  rotation by angle0 about axis a, followed by</li>
   * <li>  rotation by angle1 about axis b, followed by</li>
   * <li>  rotation by angle2 about axis c</li>
   * </ul>
   * The specification of angle0, angle1, and angle2 and axes a, b, and c
   * are determined by the enumerated constant <code>angle_type.
   * <p>
   * Because we want a transformation <em>from</em> the space coordinates
   * <em>to</em> the body coordinates, the rotation matrix components
   * actually correspond to the inverse of the transformation which
   * generated the rigid body position, i.e.,<ul>
   * <li>  rotation by -angle2 about axis c, followed by</li>
   * <li>  rotation by -angle1 about axis b, followed by</li>
   * <li>  rotation by -angle0 about axis a</li>
   * </ul>
   * <p>
   * Because of rotation matrices are orthogonal, the inverse
   * motion (from the body-centered coordinate system back to
   * the standard coordinate system) is simply the transpose of the
   * rotation matrix constructed by this constructor.</p>
   *
   * @param angle0      1st Euler angle
   * @param angle1      2nd Euler angle
   * @param angle2      3rd Euler angle
   * @param angle_type  type of Euler angles to use
   */
  void init( double angle0, double angle1, double angle2,
             EEulerAngleType angle_type = EOSAC_az_el_clock );

  /**
   * Initialize a rotation matrix based on two (non-colinear) vectors.
   * The rotation matrix is constructed by sequential application 
   * cross products.
   * <p>
   * If type == E_XY, the vectors passed in are assumed to lie in the 
   * x-y plane with v0 an x-like vector and v1 a y-like vector;
   * e_x is v0.  e_z is constructed from e_x cross v1.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YZ, the vectors passed in are assumed to lie in the
   * y-z plane with v0 an y-like vector and v1 a z-like vector;
   * e_y is v0.  e_x is constructed from e_y cross v1.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZX, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an z-like vector and v1 a x-like vector;
   * e_z is v0.  e_y is constructed from e_z cross v1.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YX, the vectors passed in are assumed to lie in the
   * x-y plane with v0 a y-like vector and v1 an x-like vector;
   * e_y is v0.  e_z is constructed from v1 cross e_y.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZY, the vectors passed in are assumed to lie in the
   * y-z plane with v0 a z-like vector and v1 a y-like vector;
   * e_z is v0.  e_x is constructed from v1 cross e_z.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_XZ, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an x-like vector and v1 a z-like vector;
   * e_x is v0.  e_y is constructed from v1 cross e_x.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   *
   * @param v0     1st vector
   * @param v1     2nd vector
   * @param type   type of input vectors
   */
  void init( dvm3_Vector const& v0,
             dvm3_Vector const& v1,
             EInputVecPair      type
           );

  /**
   * Initialize a rotation matrix based on two (non-colinear) vectors.
   * The rotation matrix is constructed by sequential application 
   * cross products.
   * <p>
   * If type == E_XY, the vectors passed in are assumed to lie in the 
   * x-y plane with v0 an x-like vector and v1 a y-like vector;
   * e_x is v0.  e_z is constructed from e_x cross v1.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YZ, the vectors passed in are assumed to lie in the
   * y-z plane with v0 an y-like vector and v1 a z-like vector;
   * e_y is v0.  e_x is constructed from e_y cross v1.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZX, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an z-like vector and v1 a x-like vector;
   * e_z is v0.  e_y is constructed from e_z cross v1.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_YX, the vectors passed in are assumed to lie in the
   * x-y plane with v0 a y-like vector and v1 an x-like vector;
   * e_y is v0.  e_z is constructed from v1 cross e_y.
   * e_x is constructed from e_y cross e_z.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_ZY, the vectors passed in are assumed to lie in the
   * y-z plane with v0 a z-like vector and v1 a y-like vector;
   * e_z is v0.  e_x is constructed from v1 cross e_z.
   * e_y is constructed from e_z cross e_x.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   * <p>
   * If type == E_XZ, the vectors passed in are assumed to lie in the
   * x-z plane with v0 an x-like vector and v1 a z-like vector;
   * e_x is v0.  e_y is constructed from v1 cross e_x.
   * e_z is constructed from e_x cross e_y.  The three vectors 
   * e_x, e_y, and e_z are normalized.
   *
   * @param v0     1st vector
   * @param v1     2nd vector
   * @param type   type of input vectors
   */
  void init( double const v0[],
             double const v1[],
             EInputVecPair      type
           );

  /**
   * Copy-assignment operator
   *
   * @return   rotation matrix
   *
   * @param rhs  value to be assigned.
   */
  dvm3_RotMat& operator=( dvm3_RotMat const& rhs );

  // /*
  //  * Assignment operator; set each element to value of rhs.
  //  */
  // dvm3_RotMat& operator=( double             rhs );

  /**
   * Invert (i.e., transpose) this rotation matrix.
   */
  void invert();

  /**
   * Test this rotation matrix for orthonormality.
   *
   * @return 1 if this is an orthonormal matrix, 0 otherwise.
   * 
   * @param tol  tolerance for determining orthonormality
   */
  int is_rotation_matrix( double const tol = 100.0 * DBL_EPSILON ) const;

  // friend std::ostream& operator<<( std::ostream& os, dvm3_RotMat const& );
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
inline dvm3_RotMat::
~dvm3_RotMat()
{}

//=========================================================================
inline dvm3_RotMat::
dvm3_RotMat( dvm3_RotMat const& other )
  : dvm3_Matrix( other )
{}

//=========================================================================
inline dvm3_RotMat& dvm3_RotMat::
operator=( dvm3_RotMat const& m )
{ data_ = m.data_;   return *this; }

#endif
