-- --8<--8<--8<--8<--
--
-- Copyright (C) 2010 Smithsonian Astrophysical Observatory
--
-- This file is part of raygen
--
-- raygen is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--
-- -->8-->8-->8-->8--

local _PACKAGE = 'saotrace.raygen.'

local raygen 	= require( 'saotrace.raygen' )
local vobj 	= require( 'saotrace.raygen.validate').vobj
local logger 	= require( 'saotrace.raygen.log').logger
local intrinsic = require( 'saotrace.raygen.intrinsic.source' )
local extent 	= require( 'saotrace.raygen.extent' )
local spectrum 	= require( 'saotrace.raygen.spectrum' )

local _M = {}

--------------------------
-- fake validation specifications used by intrinsic.  it cannot be
-- local because intrinsic uses it to create mock functions

_M.vspec = { start = 1, finish = 2 }

------------------------------------------------------------


local function is_of_class( obj, class )
   local ok, isa = pcall( obj.isa, obj, class )
   if ok and isa then
      return true, obj
   end
   return false, "not an object of type " .. class
end

local function validate_spectra( list )

   if type(list) ~= 'table' or next(list) == nil then
      return false, "expected a list of " .. _PACKAGE .. "spectrum objects"
   end

   -- if list is really a single spectrum object, wrap that in a list
   -- and continue
   if is_of_class( list, _PACKAGE .. 'spectrum' ) then
      list = { list }
   end

   -- elements of list must be raygen.spectrum objects
   for k,v in pairs(list) do

      if not is_of_class( v, _PACKAGE .. 'spectrum' ) then
	 return false,
	    "spectrum element " .. k .. " is not a " .. _PACKAGE .. "spectrum object"
      end

   end

   return true, list

end




function _M.create ( ... )

   local spec = { { name = "name" },
		  { name = "opts",
		    optional = true,
		    default = { },
		    vtable = {
		       extent = {
			  vfunc = function( arg )
				     if arg == nil then
					return true, nil
				     else
					return is_of_class( arg, 'saotrace.raygen.extent' )
				     end
				  end,
			  optional = true,
		       },

		       spectrum = {

			  vfunc = validate_spectra,
			  optional = true
		       },

		    }
		 }
	       }

   local ok, args = vobj:validate( spec, ... )

   if not ok  then
      error( "error creating source:" .. args )
   end

   local src = {}

   src.name = args.name

   -- create defaults for optional arguments which are difficult
   -- to do otherwise

   src.extent = args.opts.extent or extent.default{ name = src.name }

   src.spectrum = args.opts.spectrum or { spectrum.default{ name = src.name } }

   logger:debug( src )

   intrinsic.start{ name = src.name };
   src.extent:register()

   for _,v in pairs( src.spectrum  ) do
      v:register()
   end
   intrinsic.finish{ name = src.name };

   return src

end

return _M
